/*
 * MyEtchedBorder.java - This file is part of VSTrade.
 *
 * Copyright (C) 2007 Niklas Kyster Rasmussen
 *
 * VSTrade is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * VSTrade is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VSTrade; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * FILE DESCRIPTION:
 * Statusbar border (draw EtchedBorder on the top, only)
 */

package vstrade.shared;

//Java
import java.awt.Graphics;
import java.awt.Insets;
import java.awt.Color;
import java.awt.Component;
import javax.swing.border.AbstractBorder;

public class StatusEtchedBorder extends AbstractBorder{
	
   /** Raised etched type. */
    public static final int RAISED  = 0;
    /** Lowered etched type. */
    public static final int LOWERED = 1;

    protected int etchType;
    protected Color highlight;
    protected Color shadow;
	
	boolean bTopOnly = false;
/*
	int x;
	int y;
	int width;
	int height;
 */
    /**
     * Creates a lowered etched border whose colors will be derived
     * from the background color of the component passed into 
     * the paintBorder method.
     */
    public StatusEtchedBorder()    {
        this(LOWERED);
    }

	public StatusEtchedBorder(boolean bTopInput)    {
		bTopOnly = bTopInput;
		this.etchType = LOWERED;
    }
    /**
     * Creates an etched border with the specified etch-type
     * whose colors will be derived
     * from the background color of the component passed into 
     * the paintBorder method.
     * @param etchType the type of etch to be drawn by the border
     */
    public StatusEtchedBorder(int etchType)    {
        this(etchType, null, null);
    }

    /**
     * Creates a lowered etched border with the specified highlight and
     * shadow colors.
     * @param highlight the color to use for the etched highlight
     * @param shadow the color to use for the etched shadow
     */
    public StatusEtchedBorder(Color highlight, Color shadow)    {
        this(LOWERED, highlight, shadow);
    }

    /**
     * Creates an etched border with the specified etch-type,
     * highlight and shadow colors.
     * @param etchType the type of etch to be drawn by the border
     * @param highlight the color to use for the etched highlight
     * @param shadow the color to use for the etched shadow
     */
    public StatusEtchedBorder(int etchType, Color highlight, Color shadow)    {
        this.etchType = etchType;
        this.highlight = highlight;
        this.shadow = shadow;
    }

	/**
     * Paints the border for the specified component with the 
     * specified position and size.
     * @param c the component for which this border is being painted
     * @param g the paint graphics
     * @param x the x position of the painted border
     * @param y the y position of the painted border
     * @param width the width of the painted border
     * @param height the height of the painted border
     */
	@Override
    public void paintBorder(Component c, Graphics g, int x, int y, int width, int height) {
		int w = width;
		int h = height;

		g.translate(x, y);

		g.setColor(etchType == LOWERED? getShadowColor(c) : getHighlightColor(c));
		if (bTopOnly) {
			g.drawLine(0, 0, w, 0);
		} else { //Untested!!
			g.drawLine(0, 0, w, 0);//TOP
			//g.drawLine(w-2, 1, w-2, h);//RIGTH
		}

		
		g.setColor(etchType == LOWERED? getHighlightColor(c) : getShadowColor(c)); //White
		if (bTopOnly) {
			g.drawLine(0, 1, w, 1);
		} else {
			g.drawLine(0, 1, w, 1);//TOP
			//g.drawLine(w-1, 1, w-1, h);//RIGTH
			
		}

		g.translate(-x, -y);
    }

    /**
     * Returns the insets of the border.
     * @param c the component for which this border insets value applies
     */
	@Override
    public Insets getBorderInsets(Component c)       {
		if (bTopOnly) {
			return new Insets(2, 0, 0, 0);
		}else {
			return new Insets(2, 0, 0, 2);
		}
        
    }

    /** 
     * Reinitialize the insets parameter with this Border's current Insets. 
     * @param c the component for which this border insets value applies
     * @param insets the object to be reinitialized
     */
	@Override
    public Insets getBorderInsets(Component c, Insets insets) {
        insets.left = insets.top = insets.right = insets.bottom = 2;
        return insets;
    }

    /**
     * Returns whether or not the border is opaque.
     */
	@Override
    public boolean isBorderOpaque() { return true; }

    /**
     * Returns which etch-type is set on the etched border.
     */
    public int getEtchType() {
        return etchType;
    }

    /**
     * Returns the highlight color of the etched border
     * when rendered on the specified component.  If no highlight
     * color was specified at instantiation, the highlight color
     * is derived from the specified component's background color.
     * @param c the component for which the highlight may be derived
     */
    public Color getHighlightColor(Component c)   {
        return highlight != null? highlight : 
                                       c.getBackground().brighter();
    }

    /**
     * Returns the highlight color of the etched border.
     * Will return null if no highlight color was specified
     * at instantiation.
     */
    public Color getHighlightColor()   {
        return highlight;
    }

    /**
     * Returns the shadow color of the etched border
     * when rendered on the specified component.  If no shadow
     * color was specified at instantiation, the shadow color
     * is derived from the specified component's background color.
     * @param c the component for which the shadow may be derived
     */
    public Color getShadowColor(Component c)   {
        return shadow != null? shadow : c.getBackground().darker();
    }

    /**
     * Returns the shadow color of the etched border.
     * Will return null if no shadow color was specified
     * at instantiation.
     */
    public Color getShadowColor()   {
        return shadow;
    }
	
}
