/*
 * GUIFactory.java - This file is part of VSTrade.
 *
 * Copyright (C) 2007 Niklas Kyster Rasmussen
 *
 * VSTrade is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * VSTrade is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VSTrade; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * FILE DESCRIPTION:
 * GUI Factory. Only contains static methods. Used to create standardized swing components.
 * Allow for easy width/height changes of standardized components.
 * Almost all components in VSTrade is created by GUIFactory
 */

package vstrade.shared;

//Java
import java.awt.Dimension;
import java.awt.Insets;
import java.awt.event.ActionListener;
import java.awt.event.KeyListener;
import java.awt.event.MouseListener;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.ListModel;
import javax.swing.border.TitledBorder;

//VSTrade

public class GUIFactory {
	
	private static final int BUTTON_WIDTH = 100;
	private static final int BUTTON_SMALL_WIDTH = 44;
	private static final int SPACE_Y = 5;
	private static final int SPACE_X = 10;
	private static final int FIELD_MINIMUM_LENGTH = 250;
	
	public static final int FIELD_WIDTH = 250;
	public static final int FIELD_HEIGTH = 25;
	public static final int FIELD_WIDTH_AND_BUTTON = 16 + (FIELD_WIDTH * 2) - (BUTTON_WIDTH + 10);
	public static final int FIELD_WIDTH_X2 = (FIELD_WIDTH*2)+15;
	public static final int FIELD_WIDTH_X3 = (FIELD_WIDTH*3)+30;
	
	private static final int SPACE_BUTTON_Y = 5;
	
	public static final int LINE_AXIS = BoxLayout.LINE_AXIS;
	public static final int PAGE_AXIS = BoxLayout.PAGE_AXIS;
	
	private GUIFactory() {
		//this.jPanelR
		//JPanel.set
	}
	
	/**
	 * Creats a JPanel with the <code>components</code> side by side and aligned to the right.<br />
	 * The JPanel is as width as possible and the hight of the <code>components</code>
	 *
	 * @param components    The JComponent to add to JPanel
	 * @return              JPanel with the parsed JComponent
	 */
	public static JPanel jPRow(JComponent... jComponents){
		return jPCostum(jComponents, LINE_AXIS, false, false, 0, 0, true, false, null);
	}
	/**
	 * Creats a JPanel with the <code>components</code> bellow each other and the size of the <code>components</code>.<br>
	 * 
	 * @param components    The JComponent to add to JPanel
	 * @return              JPanel with the parsed JComponent
	 */
	public static JPanel jPColumn(JComponent... jComponents){
		return jPCostum(jComponents, PAGE_AXIS, false, false, 0, 0, false, false, null);
	}
	//Dimension JPanel
	/**
	 * Creats a JPanel with the <code>components</code> bellow each other, aligned at the top, and the size of <code>dimension</code>.<br>
	 * 
	 * @param dimension     The Dimension of the created JPanel
	 * @param components    The JComponent to add to JPanel
	 * @return              JPanel with the parsed JComponent
	 */
	public static JPanel jPDimension(Dimension dimension, JComponent... jComponents){
		return jPCostum(jComponents, PAGE_AXIS, false, false, 0, 0, false, false, dimension);
	}
//Title JPanel
	public static JPanel jPTitle(JComponent jComponent){
		return jPTitle(GUIFactory.LINE_AXIS, jComponent);
	}
	public static JPanel jPTitle(int nAxis, JComponent... jComponents){
		JPanel jPanel = new JPanel();
		TitledBorder titledBorder = BorderFactory.createTitledBorder("[NOT TRASNLATED]");
		//Calculate offset
		Insets insets = titledBorder.getBorderInsets(jPanel);
		Dimension d = titledBorder.getMinimumSize(jPanel);
		int x = insets.left+insets.right;
		int y = (int) d.getHeight();
		
		jPanel = jPCostum(jComponents, nAxis, true, false, x, y, false, false, null);
		jPanel.setBorder( titledBorder );
		return jPanel;
	}
//Button JPanel
	public static JPanel jPButton(JComponent... jComponents){
		return jPCostum(jComponents, LINE_AXIS, false, true, 0, 0, false, false, null);
	}
	/**
	 *
	 * @param	components	Vector with the JComponent's to add to the JPanel
	 * @param	nAxis		The axis to add JComponent's on.<br />LINE_AXIS (Side by Side)<br />PAGE_AXIS (Bellow each other)
	 * @param	bSpace		Add space between JComponent's
	 * @param	xOffset		Width added to the JPanel besides the JComponent's width
	 * @param	yOffset		Hight added to the JPanel besides the JComponent's hight
	 * @param	xMax		Add width as possible
	 * @param	yMax		Add high as possible
	 * @param	d			The Dimension of the JPanel (Overwrites: xOffset, yOffset, xMax, yMax)
	 *
	 * @return		
	 */
	private static JPanel jPCostum(JComponent[] jComponents, int nAxis, boolean bSpace, boolean bCustomSpace, int xOffset, int yOffset, boolean xMax, boolean yMax, Dimension d){
		int y = 0;
		int x = 0;
		int xCustomSpace = 0;
		int yCustomSpace = 0;
		JPanel jPanel = new JPanel();
		
		jPanel.setLayout( new BoxLayout(jPanel, nAxis) ); 
		
		Dimension temp;
		
		if (bCustomSpace && jComponents.length > 1) {
			
			for (int a = 0; a < jComponents.length; a++){
				temp = jComponents[a].getPreferredSize();
				if (nAxis == PAGE_AXIS){
					yCustomSpace = yCustomSpace + temp.height;
				}
				if (nAxis == LINE_AXIS){
					xCustomSpace = xCustomSpace + temp.width;
				}
				
			}
			xCustomSpace = (FIELD_WIDTH - xCustomSpace) / (jComponents.length - 1) ;
		}
		
		
		
		for (int a = 0; a < jComponents.length; a++){
			temp = jComponents[a].getPreferredSize();
			if (nAxis == PAGE_AXIS){
				y = y + temp.height;
				if (bSpace && a > 0){
					jPanel.add(Box.createRigidArea( new Dimension(FIELD_WIDTH, SPACE_Y)) );
					y = y + SPACE_Y;
				}
				if (bCustomSpace && a > 0){
					jPanel.add(Box.createRigidArea( new Dimension(FIELD_WIDTH, yCustomSpace)) );
					y = y + yCustomSpace;
				}
			} else {
				if (temp.height > y) y = temp.height;
			}
			if (nAxis == LINE_AXIS){
				x = x + temp.width;
				if (bSpace && a > 0){
					jPanel.add(Box.createRigidArea( new Dimension(SPACE_X, FIELD_HEIGTH)) );
					x = x + SPACE_X;
				}
				if (bCustomSpace && a > 0){
					jPanel.add(Box.createRigidArea( new Dimension(xCustomSpace, FIELD_HEIGTH)) );
					x = x + xCustomSpace;
				}
			} else {
				if (temp.width > x) x = temp.width;
			}
		
			jPanel.add(jComponents[a]);
		}
		x = x + xOffset;
		y = y + yOffset;
		if (xMax) x = Short.MAX_VALUE;
		if (yMax) y = Short.MAX_VALUE;
		if (d == null) d = new Dimension(x, y);
		jPanel.setPreferredSize( d );
		jPanel.setMaximumSize( d );
		
		return jPanel;
	}
//JScrollPane
	public static JScrollPane jScrollPanel(JComponent jComponent, int y){
		JScrollPane jScrollPanel = new JScrollPane(jComponent);
		jScrollPanel.setPreferredSize( new Dimension( FIELD_WIDTH, FIELD_HEIGTH*y) );
		jScrollPanel.setMinimumSize( new Dimension( FIELD_MINIMUM_LENGTH, FIELD_HEIGTH*y) );
		return jScrollPanel;
	}
	private static JScrollPane jScrollPanel(JComponent jComponent, int x, int y){
		JScrollPane jScrollPanel = new JScrollPane(jComponent);
		jScrollPanel.setPreferredSize( new Dimension( FIELD_WIDTH*x, FIELD_HEIGTH*y) );
		jScrollPanel.setMinimumSize( new Dimension( FIELD_MINIMUM_LENGTH*x, FIELD_HEIGTH*y) );
		return jScrollPanel;
	}
	public static JPanel jPRadioButton(JRadioButton jRadioButton, JLabel jLabel){
		return jPCheckRadioBox(jRadioButton, jLabel);
	}
//JCheckBox JPanel
	public static JPanel jPCheckBox(JCheckBox jCheckBox, JLabel jLabel){
		/*
		JPanel panel = new JPanel();
		panel.add(checkBox);
		panel.add(label);
		panel.setLayout( new BoxLayout(panel, BoxLayout.LINE_AXIS) ); 
		panel.setMaximumSize( new Dimension( FIELD_WIDTH, FIELD_HEIGTH ) );
		panel.setPreferredSize( new Dimension( FIELD_WIDTH, FIELD_HEIGTH ) );
		panel.setMinimumSize( new Dimension( FIELD_MINIMUM_LENGTH, FIELD_HEIGTH ) );
		return panel;
		 */
		return jPCheckRadioBox(jCheckBox, jLabel);
	}
	private static JPanel jPCheckRadioBox(JComponent jComponent, JLabel jLabel){
		JPanel jPanel = new JPanel();
		jPanel.add(jComponent);
		jPanel.add(jLabel);
		jPanel.setLayout( new BoxLayout(jPanel, BoxLayout.LINE_AXIS) ); 
		jPanel.setMaximumSize( new Dimension( FIELD_WIDTH, FIELD_HEIGTH ) );
		jPanel.setPreferredSize( new Dimension( FIELD_WIDTH, FIELD_HEIGTH ) );
		jPanel.setMinimumSize( new Dimension( FIELD_MINIMUM_LENGTH, FIELD_HEIGTH ) );
		return jPanel;
	}
//JLabel
	public static JLabel jLabel(MouseListener mouseListener){
		return jLabel(0, mouseListener);
	}
	private static JLabel jLabel(int x, MouseListener mouseListener){
		JLabel jLabel = new JLabel();
		jLabel.addMouseListener(mouseListener);
		if (x > 0){
			jLabel.setPreferredSize( new Dimension(FIELD_WIDTH*x, FIELD_HEIGTH) );
			jLabel.setMinimumSize( new Dimension(FIELD_MINIMUM_LENGTH*x, FIELD_HEIGTH) );
		}
		return jLabel;
	}
//JTextField
	public static JTextField jTextField(KeyListener textListener){
		JTextField jTextField = new JTextField();
		jTextField.addKeyListener(textListener);
		jTextField.setPreferredSize( new Dimension(FIELD_WIDTH, FIELD_HEIGTH) );
		jTextField.setMinimumSize( new Dimension(FIELD_MINIMUM_LENGTH, FIELD_HEIGTH) );
		//text.setPreferredSize( new Dimension(MAIN_FIELD_LENGTH, MAIN_FIELD_HEIGTH) );
		return jTextField;
	}
//JComboBox
	public static JComboBox jComboBox(){
		return jComboBox(null, null, "", null, null);
	}
	public static JComboBox jComboBox(ActionListener actionListener, String sActionCommand){
		return jComboBox(null, actionListener, sActionCommand, null, null);
	}
	public static JComboBox jComboBox(ActionListener actionListener, String sActionCommand, KeyListener keyListener){
		return jComboBox(null, actionListener, sActionCommand, keyListener, null);
	}
	public static JComboBox jComboBox(Object[] array, ActionListener actionListener, String sActionCommand){
		return jComboBox(array, actionListener, sActionCommand, null, null);
	}
	public static JComboBox jComboBoxSmall(ActionListener actionListener, String sActionCommand){
		return jComboBoxSmall(actionListener, sActionCommand, null);
	}
	public static JComboBox jComboBoxSmall(ActionListener actionListener, String sActionCommand, KeyListener keyListener){
		JComboBox jComboBox = new JComboBox();
		//combo.setMaximumSize( new Dimension(FIELD_WIDTH - (BUTTON_SMALL_WIDTH + SPACE_X + 5), FIELD_HEIGTH) );
		jComboBox.setPreferredSize( new Dimension(FIELD_WIDTH - (BUTTON_SMALL_WIDTH + SPACE_X), FIELD_HEIGTH) );
		jComboBox.setMinimumSize( new Dimension(FIELD_MINIMUM_LENGTH - (BUTTON_SMALL_WIDTH + SPACE_X + 5), FIELD_HEIGTH) );
		jComboBox.setActionCommand(sActionCommand);
		jComboBox.addActionListener(actionListener);
		jComboBox.addKeyListener(keyListener);
		return jComboBox;
	}
	private static JComboBox jComboBox(Object[] array, ActionListener actionListener, String sActionCommand, KeyListener keyListener, MouseListener mouseListener){
		JComboBox jComboBox;
		if (array != null){
			jComboBox = new JComboBox(array);
		} else {
			jComboBox = new JComboBox();
		}
		jComboBox.setMaximumSize( new Dimension(FIELD_WIDTH, FIELD_HEIGTH) );
		jComboBox.setPreferredSize( new Dimension(FIELD_WIDTH, FIELD_HEIGTH) );
		jComboBox.setMinimumSize( new Dimension(FIELD_MINIMUM_LENGTH, FIELD_HEIGTH) );
		jComboBox.setActionCommand(sActionCommand);
		jComboBox.addActionListener(actionListener);
		jComboBox.addKeyListener(keyListener);
		jComboBox.addMouseListener(mouseListener);
		return jComboBox;
	}
//JCheckBox
	public static JCheckBox jCheckBox(){
		return jCheckBox(null, "");
	}
	public static JCheckBox jCheckBox(ActionListener actionListener, String sActionCommand){
		JCheckBox jCheckBox = new JCheckBox();
		jCheckBox.setActionCommand(sActionCommand);
		jCheckBox.addActionListener(actionListener);
		//check.setMinimumSize( new Dimension(Values.MAIN_FIELD_LENGTH, 25) );
		//check.setPreferredSize( new Dimension(Values.MAIN_FIELD_LENGTH, 25) );
		return jCheckBox;
	}
	public static JRadioButton jRadioButton(){
		return jRadioButton(null, "");
	}
	public static JRadioButton jRadioButton(ActionListener actionListener, String sActionCommand){
		JRadioButton jRadioButton = new JRadioButton();
		jRadioButton.setActionCommand(sActionCommand);
		jRadioButton.addActionListener(actionListener);
		//check.setMinimumSize( new Dimension(Values.MAIN_FIELD_LENGTH, 25) );
		//check.setPreferredSize( new Dimension(Values.MAIN_FIELD_LENGTH, 25) );
		return jRadioButton;
	}
	
	 
//VSTList
	public static MultiSelectionList jMultiSelectionList(){
		return jMultiSelectionList(null, null, null, null);
	}
	public static MultiSelectionList jMultiSelectionList(String[] data){
		return jMultiSelectionList(data, null, null, null);
	}
	public static MultiSelectionList jMultiSelectionList(ListModel listModel, KeyListener keyListener, MouseListener mouseListener){
		return jMultiSelectionList(null, listModel, keyListener, mouseListener);
	}
	
	
	private static MultiSelectionList jMultiSelectionList(String[] data, ListModel listModel, KeyListener keyListener, MouseListener mouseListener){
		MultiSelectionList multiSelectionList = new MultiSelectionList();
		if (keyListener != null) multiSelectionList.addKeyListener(keyListener);
		if (mouseListener != null) multiSelectionList.addMouseListener(mouseListener);
		if (listModel != null) multiSelectionList.setModel(listModel);
		if (data != null) multiSelectionList.setListData(data);
		return multiSelectionList;
	}
//JButton
	public static JButton jButton(ActionListener actionListener, String sActionCommand){
		JButton jButton = new JButton();
		jButton.setMaximumSize( new Dimension(BUTTON_WIDTH, FIELD_HEIGTH) );
		jButton.setPreferredSize( new Dimension(BUTTON_WIDTH, FIELD_HEIGTH) );
		jButton.setMinimumSize( new Dimension(BUTTON_WIDTH, FIELD_HEIGTH) );
		jButton.setActionCommand(sActionCommand);
		jButton.addActionListener( actionListener );
		return jButton;
	}
	public static JButton jButtonSmall(ActionListener actionListener, String sActionCommand){
		JButton jButton = new JButton();
		jButton.setMaximumSize( new Dimension(BUTTON_SMALL_WIDTH, FIELD_HEIGTH) );
		jButton.setPreferredSize( new Dimension(BUTTON_SMALL_WIDTH, FIELD_HEIGTH) );
		jButton.setMinimumSize( new Dimension(BUTTON_SMALL_WIDTH, FIELD_HEIGTH) );
		jButton.setActionCommand(sActionCommand);
		jButton.addActionListener( actionListener );
		return jButton;
	}
	public static void setBorderTitle(JPanel jPanel, String sTitle){
		((TitledBorder)jPanel.getBorder()).setTitle(sTitle);
	}
	/*
	public static JPanel JPanel(int nAxis, JComponent... components){
		int y = 0;
		int x = 0;
		JPanel panel = new JPanel();

		Dimension temp;
		
		for (int a = 0; a < components.length; a++){
			temp = components[a].getPreferredSize();
			if (nAxis == PAGE_AXIS){
				y = y + temp.height;
			} else {
				if (temp.height > y) y = temp.height;
			}
			if (nAxis == LINE_AXIS){
				x = x + temp.width;
			} else {
				if (temp.width > x) x = temp.width;
			}
		
			panel.add(components[a]);
		}
		Dimension d = new Dimension(x, y);
		panel.setPreferredSize( d );
		panel.setMaximumSize( d );
		
		return panel;
	}
	 **/
}	


