/*
 * TradeTableRow.java - This file is part of VSTrade.
 *
 * Copyright (C) 2007 Niklas Kyster Rasmussen
 *
 * VSTrade is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * VSTrade is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VSTrade; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * FILE DESCRIPTION:
 * Used by the trade table (part of the Glazed Lists implementation)
 */

package vstrade.moduls.trade;

//VSTrade
import vstrade.data.bases.Base;
import vstrade.data.bases.Cargo;

public class TradeTableRow implements Comparable<TradeTableRow>{

	private Base buyBase = null;
	private Cargo buyCargo = null;
	private Base sellBase = null;
	private Cargo sellCargo = null;
	
	private String sBuyBaseName;
	private String sSellBaseName;
	private String sCargoName;
	
	private int nQuantityMax;
	private int nQuantityAvg;
	private int nQuantityMin;
	private float nCargoVolume;
	private int nPercentProfitMax;
	private int nPercentProfitAvg;
	private int nPercentProfitMin;
	private int nCashProfitMax;
	private int nCashProfitAvg;
	private int nCashProfitMin;
	
	/** Creates a new instance of TradeTableRow */
	public TradeTableRow(Base buyBase, Cargo buyCargo, Base sellBase, Cargo sellCargo, boolean bUpgrade){
		this.buyBase = buyBase;
		this.buyCargo = buyCargo;
		this.sellBase = sellBase;
		this.sellCargo = sellCargo;

		sBuyBaseName = buyBase.getName();
		sCargoName = buyCargo.getName();
		sSellBaseName = sellBase.getName();
		
		
		nQuantityMax = buyCargo.getQuantityMaximum();
		nQuantityAvg = buyCargo.getQuantityAverage();
		nQuantityMin = buyCargo.getQuantityMinimum();
		nCargoVolume = buyCargo.getCargoVolume();
		nPercentProfitMax = Cargo.calcPercentProfitMax(buyCargo, sellCargo, bUpgrade);
		nPercentProfitAvg = Cargo.calcPercentProfitAvg(buyCargo, sellCargo, bUpgrade);
		nPercentProfitMin = Cargo.calcPercentProfitMin(buyCargo, sellCargo, bUpgrade);
		nCashProfitMax = Cargo.calcCashProfitMax(buyCargo, sellCargo, bUpgrade);
		nCashProfitAvg = Cargo.calcCashProfitAvg(buyCargo, sellCargo, bUpgrade);
		nCashProfitMin = Cargo.calcCashProfitMin(buyCargo, sellCargo, bUpgrade);
	}
	public Base getBuyBase(){
		return buyBase;
	}
	
	public Base getSellBase(){
		return sellBase;
	}
	
	public Cargo getCargo(){
		return buyCargo;
	}
	
	public String getBuyBaseName(){
		return sBuyBaseName;
	}
	
	public String getSellBaseName(){
		return sSellBaseName;
	}
	
	public String getCargoName(){
		return sCargoName;
	}
	
	public int getQuantityMax() {
		return nQuantityMax;
	}

	public int getQuantityAvg() {
		return nQuantityAvg;
	}

	public int getQuantityMin() {
		return nQuantityMin;
	}
	
	public float getCargoVolume(){
		return nCargoVolume;
	}
	
	public int getPercentProfitMax() {
		return nPercentProfitMax;
	}
	
	public int getPercentProfitAvg() {
		return nPercentProfitAvg;
	}

	public int getPercentProfitMin() {
		return nPercentProfitMin;
	}

	public int getCashProfitMax() {
		return nCashProfitMax;
	}

	public int getCashProfitAvg() {
		return nCashProfitAvg;
	}

	public int getCashProfitMin() {
		return nCashProfitMin;
	}
	
	@Override
	public int compareTo(TradeTableRow o) {
		return o.getBuyBase().getName().compareTo(this.getBuyBase().getName());
	}
}