/*
 * TradeTableFormat.java - This file is part of VSTrade.
 *
 * Copyright (C) 2007 Niklas Kyster Rasmussen
 *
 * VSTrade is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * VSTrade is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VSTrade; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * FILE DESCRIPTION:
 * TableFormat, used by the trade table (part of the Glazed Lists implementation)
 */

package vstrade.moduls.trade;

//Java
import java.util.Comparator;

//Glazed Lists
import ca.odell.glazedlists.gui.AdvancedTableFormat;

//VSTrade
import vstrade.Program;
import vstrade.shared.FloatComparator;
import vstrade.shared.IntegerComparator;
import vstrade.shared.StringComparator;

public class TradeTableFormat implements AdvancedTableFormat<TradeTableRow> {
	
	private Comparator integerComparator;
	private Comparator stringComparator;
	private Comparator floatComparator;
	
	private Program program;
	
	/** Creates a new instance of TradeTableFormat */
	public TradeTableFormat(Program program) {
		this.program = program;
		
		integerComparator = new IntegerComparator();
		stringComparator = new StringComparator();
		floatComparator = new FloatComparator();
	}
	
	@Override
	public int getColumnCount() {
		return program.getSettings().getTradeSettings().getTradeTableColumns().size();
	}
	
	@Override
	public String getColumnName(int i) {
		return program.getSettings().getLanguageSettings().getString(program.getSettings().getTradeSettings().getTradeTableColumn(i), "trade");
		
	}
	
	@Override
	public Object getColumnValue(TradeTableRow row, int i) {
		String sColumn = program.getSettings().getTradeSettings().getTradeTableColumn(i);
		if (sColumn.equals("TableBuyBase")) return row.getBuyBaseName();
		if (sColumn.equals("TableSellBase")) return row.getSellBaseName();
		if (sColumn.equals("TableCommerce")) return row.getCargoName();
		if (sColumn.equals("TableMaxQuantity")) return row.getQuantityMax();
		if (sColumn.equals("TableAvgQuantity")) return row.getQuantityAvg();
		if (sColumn.equals("TableMinQuantity")) return row.getQuantityMin();
		if (sColumn.equals("TableVolume")) return row.getCargoVolume();
		if (sColumn.equals("TableMaxPercent")) return row.getPercentProfitMax();
		if (sColumn.equals("TableAvgPercent")) return row.getPercentProfitAvg();
		if (sColumn.equals("TableMinPercent")) return row.getPercentProfitMin();
		if (sColumn.equals("TableMaxCash")) return row.getCashProfitMax();
		if (sColumn.equals("TableAvgCash")) return row.getCashProfitAvg();
		if (sColumn.equals("TableMinCash")) return row.getCashProfitMin();
		return new String();
	}
	
	@Override
	public Class getColumnClass(int i) {
		String sColumn = program.getSettings().getTradeSettings().getTradeTableColumn(i);
		if (sColumn.equals("TableBuyBase")) return String.class;
		if (sColumn.equals("TableSellBase")) return String.class;
		if (sColumn.equals("TableCommerce")) return String.class;
		if (sColumn.equals("TableMaxQuantity")) return Integer.class;
		if (sColumn.equals("TableAvgQuantity")) return Integer.class;
		if (sColumn.equals("TableMinQuantity")) return Integer.class;
		if (sColumn.equals("TableVolume")) return Float.class;
		if (sColumn.equals("TableMaxPercent")) return Integer.class;
		if (sColumn.equals("TableAvgPercent")) return Integer.class;
		if (sColumn.equals("TableMinPercent")) return Integer.class;
		if (sColumn.equals("TableMaxCash")) return Integer.class;
		if (sColumn.equals("TableAvgCash")) return Integer.class;
		if (sColumn.equals("TableMinCash")) return Integer.class;
		return Object.class;
	}
	
	@Override
	public Comparator getColumnComparator(int i) {
		String sColumn = program.getSettings().getTradeSettings().getTradeTableColumn(i);
		if (sColumn.equals("TableBuyBase")) return stringComparator;
		if (sColumn.equals("TableSellBase")) return stringComparator;
		if (sColumn.equals("TableCommerce")) return stringComparator;
		if (sColumn.equals("TableMaxQuantity")) return integerComparator;
		if (sColumn.equals("TableAvgQuantity")) return integerComparator;
		if (sColumn.equals("TableMinQuantity")) return integerComparator;
		if (sColumn.equals("TableVolume")) return floatComparator;
		if (sColumn.equals("TableMaxPercent")) return integerComparator;
		if (sColumn.equals("TableAvgPercent")) return integerComparator;
		if (sColumn.equals("TableMinPercent")) return integerComparator;
		if (sColumn.equals("TableMaxCash")) return integerComparator;
		if (sColumn.equals("TableAvgCash")) return integerComparator;
		if (sColumn.equals("TableMinCash")) return integerComparator;
		return null;
	}
}
