/*
 * TradeSearch.java - This file is part of VSTrade.
 *
 * Copyright (C) 2007 Niklas Kyster Rasmussen
 *
 * VSTrade is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * VSTrade is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VSTrade; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * FILE DESCRIPTION:
 * Find profitable commerce to trade between bases and planets
 */

package vstrade.moduls.trade;

//Java
import javax.swing.JButton;
import java.util.List;
import java.util.Vector;

//Glazed Lists
import ca.odell.glazedlists.EventList;

//VSTrade
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.IdentityHashMap;
import java.util.Map;
import java.util.Set;
import vstrade.Program;
import vstrade.io.Log;
import vstrade.data.bases.Base;
import vstrade.data.bases.Cargo;
import vstrade.data.Node;

public class TradeSearch extends Thread {
	
	public final static String BUY_CARGO = "BuyCargo";
	public final static String SELL_CARGO = "SellCargo";
	public final static String BUY_BASE = "BuyBase";
	public final static String SELL_BASE = "SellBase";

	private final int MIN_QUANTITY = 1;
	
	private Vector<Node> buyBases;
	private Vector<Node> sellBases;
	private Base buyBase;
	private Base sellBase;
	private int nMinimumPercentProfit;
	private int nMinimumCashProfit;
	private boolean bUpgrade;
	private boolean bRisk;
	private boolean bBestTrade;
	private boolean bBestCargoTrade;
	private List<TradeTableRow> result;
	private JButton jClear;
	private JButton jSearch;
	
	private Program program;
	private EventList<TradeTableRow> tradeEventList;
	
	/** Creates a new instance of TradeSearch */
	public TradeSearch(Program program, JButton jClear, JButton jSearch, EventList<TradeTableRow> tradeEventList, Vector<Node> buyBases, Vector<Node> sellBases, int nMinimumPercentProfit, int nMinimumCashProfit, boolean bRisk, boolean bBestTrade, boolean bBestCargoTrade) {
		this.program = program;
		this.jClear = jClear;
		this.jSearch = jSearch;
		this.tradeEventList = tradeEventList;
		this.buyBases = buyBases;
		this.sellBases = sellBases;
		this.nMinimumPercentProfit = nMinimumPercentProfit;
		this.nMinimumCashProfit = nMinimumCashProfit;
		this.bRisk = bRisk;
		this.bBestTrade = bBestTrade;
		this.bBestCargoTrade = bBestCargoTrade;
		this.setName("TradeSearch");
	}
	
	@Override
	public void run() {
		jClear.setEnabled(false);
		jSearch.setEnabled(false);
		result = new Vector<TradeTableRow>();
		int aMax = buyBases.size();
		int bMax = sellBases.size();
		for (int a = 0; a < buyBases.size(); a++){
			buyBase = (Base) buyBases.get(a);
			for (int b = 0; b < sellBases.size(); b++){
				program.getStatusView().setTradeSearchProgress( calcProgress(aMax, bMax, a, b) );
				sellBase = (Base) sellBases.get(b);
				deepSearch(buyBase, sellBase);
			}
		}
		
		if (bBestCargoTrade) {
 			bestCargoTrade();
 		}
 		else if (bBestTrade) {
 			bestTrade();
 		}
		program.getStatusView().setTradeResults(result.size());
	
		tradeEventList.getReadWriteLock().writeLock().lock();
		tradeEventList.clear();
		if (result.size() > 0){ //Nothing found
			tradeEventList.addAll(result);
		}
		tradeEventList.getReadWriteLock().writeLock().unlock();
		jClear.setEnabled(true);
		jSearch.setEnabled(true);
		program.getStatusView().setTradeSearchProgress(100);
		return;
	}
	
	private void deepSearch(Node buyNode, Node sellNode){
		//Get Base's Keys
		Vector<Node> buyNodes = buyNode.getValues();
		Vector<Node> sellNodes = sellNode.getValues();
		for (int a = 0; a < buyNodes.size() && a < sellNodes.size(); a++){
			//Get each child of the buyNode and sellNode
			Node buy = buyNodes.get(a);
			Node sell = sellNodes.get(a);
			if (buy.getParent() instanceof Base && buy.getName().equals("upgrades")){
				bUpgrade = true;
			}
			if (buy.getParent() instanceof Base && !buy.getName().equals("upgrades")){
				bUpgrade = false;
			}
			if (buy instanceof Cargo && sell instanceof Cargo){ //Cargo
				compare((Cargo) buy, (Cargo) sell); //Preforme the comparision
			} else { //Category
				deepSearch(buy, sell); //Continue the search
			}
		}
		
	}
	private void compare(Cargo buyCargo, Cargo sellCargo){
		if (!buyCargo.getName().equals(sellCargo.getName()) ){
			Log.error(this.getClass()+".compare() :: Search out of sync (Buy: "+buyCargo.getName()+" Sell: "+sellCargo.getName());
		}
		int nBuyPriceAvg;
		int nBuyQuantityAvg;
		int nSellPriceAvg;
		int nPercentProfitAvg;
		int nCashProfitAvg;
		int nCashProfitMin;
		nBuyPriceAvg = buyCargo.getPriceAverage();
		nBuyQuantityAvg = buyCargo.getQuantityAverage();
		nSellPriceAvg = sellCargo.getPriceAverage();
		if (nBuyPriceAvg > 0 && nSellPriceAvg > 0){ //Not zero (Can't divide by zero)
			nPercentProfitAvg = Cargo.calcPercentProfitAvg(buyCargo, sellCargo, bUpgrade);
			nCashProfitAvg = Cargo.calcCashProfitAvg(buyCargo, sellCargo, bUpgrade);
			nCashProfitMin = Cargo.calcCashProfitMin(buyCargo, sellCargo, bUpgrade);
			if (nBuyQuantityAvg >= MIN_QUANTITY //Above minimum quantity
				&& nPercentProfitAvg >= nMinimumPercentProfit //Above Minimum Percent Profit (Entered by the user)
				&& nCashProfitAvg >= nMinimumCashProfit //Above Minimum Cash Profit (Entered by the user)
				&& (nCashProfitMin > 0 || (bRisk && nCashProfitAvg != 0) ) //Fail safe or risky
				)
			{
				addResult(buyCargo, sellCargo);
			}
		}
	}
	private void addResult(Cargo buyCargo, Cargo sellCargo){
		result.add(new TradeTableRow(buyBase, buyCargo, sellBase, sellCargo, bUpgrade) );
	}
	private int calcProgress(int aMax, int bMax, int a, int b){
		float area = aMax * bMax;
		int progress = (int) ( ( ((a * bMax) + b) / area) * 100);
		return progress;
		
	}
	private void bestTrade(){
		Base currentBase;
		Base lastBase = new Base("", "");
		TradeTableRow currentRow;
		TradeTableRow bestPercentRow = null;
		TradeTableRow bestCashRow = null;
		List<TradeTableRow> bestTradeResult = new Vector<TradeTableRow>();
		for (int a = 0; a < result.size(); a++){
			currentRow = result.get(a);
			currentBase = (Base) currentRow.getBuyBase();
			if (lastBase.equals(currentBase)){
				if (currentRow.getPercentProfitAvg() > bestPercentRow.getPercentProfitAvg()){
					bestTradeResult.remove(bestPercentRow);
					bestTradeResult.add(currentRow);
					bestPercentRow = currentRow;
				}
				if (currentRow.getCashProfitAvg() > bestPercentRow.getCashProfitAvg()){
					bestTradeResult.remove(bestCashRow);
					bestTradeResult.add(currentRow);
					bestCashRow = currentRow;
				}
			} else {
				bestTradeResult.add(currentRow);
				bestPercentRow = currentRow;
				bestCashRow = currentRow;
			}
			lastBase = currentBase;
			
		}
		result = bestTradeResult;
	}
	
 	private void bestCargoTrade(){
 		Map<String, List<TradeTableRow>> bestPercentRows = new HashMap<String, List<TradeTableRow>>();
 		Map<String, List<TradeTableRow>> bestCashRows = new HashMap<String, List<TradeTableRow>>();
 
 		// find best rows per cargo
 		// if several rows are best, take them all
 		for (int a = 0; a < result.size(); a++){
 			TradeTableRow currentRow = result.get(a);
 			Cargo currentCargo = currentRow.getCargo();
 			String currentName = currentCargo.getName();
 			
 			List<TradeTableRow> bestPercentRow = bestPercentRows.get(currentName);
 			if (bestPercentRow == null || currentRow.getPercentProfitAvg() > bestPercentRow.get(0).getPercentProfitAvg()) {
 				// no or better entry
 				bestPercentRows.put(currentName, new ArrayList<TradeTableRow>(Collections.singleton(currentRow)));
 			}
 			else if (currentRow.getPercentProfitAvg() == bestPercentRow.get(0).getPercentProfitAvg()) {
 				// entry as good as best so far
 				bestPercentRow.add(currentRow);
 			}
 			
 			List<TradeTableRow> bestCashRow = bestCashRows.get(currentName);
 			if (bestCashRow == null || currentRow.getCashProfitAvg() > bestCashRow.get(0).getCashProfitAvg()) {
 				// no or better entry
 				bestCashRows.put(currentName, new ArrayList<TradeTableRow>(Collections.singleton(currentRow)));
 			}
 			else if (currentRow.getCashProfitAvg() == bestCashRow.get(0).getCashProfitAvg()) {
 				// entry as good as best so far
 				bestCashRow.add(currentRow);
 			}
 		}
 
 		// remove identical rows
 		
 		// awkward workaround, as there is no IdentityHashSet
 		// @see http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=4479578
 		Set<TradeTableRow> rows = Collections.newSetFromMap(new IdentityHashMap<TradeTableRow, Boolean>());
 		for (List<TradeTableRow> bestRows : bestPercentRows.values()) {
 			rows.addAll(bestRows);
 		}
 		for (List<TradeTableRow> bestRows : bestCashRows.values()) {
 			rows.addAll(bestRows);
 		}
 		
 		result = new Vector<TradeTableRow>(rows);
 	}
}
