/*
 * MessageControl.java - This file is part of VSTrade.
 *
 * Copyright (C) 2007 Niklas Kyster Rasmussen
 *
 * VSTrade is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * VSTrade is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VSTrade; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * FILE DESCRIPTION:
 * The Statusbar, show progress and number of results for searches and save progress
 */

package vstrade.moduls.status;

import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.Timer;
import javax.swing.border.CompoundBorder;
import vstrade.Program;
import vstrade.shared.StatusEtchedBorder;
import vstrade.shared.ModuleView;

public class StatusView extends ModuleView implements ActionListener {
	
	private final int STATUS_HEIGHT = 25;
	
	private JLabel jTradeResult;
	private JLabel jPriceResult;
	private JLabel jInfo;
	private JProgressBar jTradeSearch;
	private JPanel jTradeSearchPanel;
	private JProgressBar jPriceSearch;
	private JPanel jPriceSearchPanel;
	private JProgressBar jSettingsSave;
	private JPanel jSettingsSavePanel;
	
	private Timer jTradeTimer;
	private Timer jPriceTimer;
	private Timer jSettingsTimer;
	
	private CompoundBorder jFieldBorder;
	
	private Program program;
	
	/** Creates a new instance of MessageControl */
	public StatusView(Program program) {
		this.program = program;
		
		StatusEtchedBorder jStatusBorder = new StatusEtchedBorder();
		jFieldBorder = BorderFactory.createCompoundBorder( BorderFactory.createMatteBorder(0, 1, 0, 0, jStatusBorder.getShadowColor(this)), BorderFactory.createMatteBorder(0, 1, 0, 0, jStatusBorder.getHighlightColor(this)));
		
		jTradeTimer = new Timer(500, this);
		jPriceTimer = new Timer(500, this);
		jSettingsTimer = new Timer(500, this);
		
		this.setPreferredSize( new Dimension(Short.MAX_VALUE, STATUS_HEIGHT) );
		this.setMaximumSize( new Dimension(Short.MAX_VALUE, STATUS_HEIGHT) );
		this.setBorder( jStatusBorder );
		this.setLayout( new BoxLayout(this, BoxLayout.LINE_AXIS) );
		
		jInfo = new JLabel();
		jInfo.setPreferredSize( new Dimension(Short.MAX_VALUE, STATUS_HEIGHT) );
		jInfo.setMaximumSize( new Dimension(Short.MAX_VALUE, STATUS_HEIGHT) );
		jInfo.setMinimumSize( new Dimension(5, STATUS_HEIGHT) );
		jInfo.setBorder( BorderFactory.createEmptyBorder(0, 5, 0, 0) );
		
		jSettingsSave = createProgress();
		jSettingsSavePanel = createProgressPanel(jSettingsSave);
		
		jTradeSearch = createProgress();
		jTradeSearchPanel = createProgressPanel(jTradeSearch);
		jTradeResult = createResult();
		
		
		jPriceSearch = createProgress();
		jPriceSearchPanel = createProgressPanel(jPriceSearch);
		jPriceResult = createResult();
		
		this.add(jInfo);
		this.add(jTradeSearchPanel);
		this.add(jPriceSearchPanel);
		this.add(jSettingsSavePanel);
		this.add(jTradeResult);
		this.add(jPriceResult);
	}
	
	@Override
	public void setData() {
		updateLanguage();
	}
	
	@Override
	public void clearData() {}
	
	@Override
	public void updateLanguage() {
		jSettingsSave.setToolTipText(program.getSettings().getLanguageSettings().getString("SettingsSaveProgress", "status"));
		jTradeSearch.setToolTipText(program.getSettings().getLanguageSettings().getString("TradeSearchProgress", "status"));
		jTradeResult.setToolTipText(program.getSettings().getLanguageSettings().getString("TradeResults", "status"));
		jPriceSearch.setToolTipText(program.getSettings().getLanguageSettings().getString("PriceSearchProgress", "status"));
		jPriceResult.setToolTipText(program.getSettings().getLanguageSettings().getString("PriceResults", "status"));
	}
	public void setStatusbarText(String sText){
		jInfo.setText(sText);
	}
	public void setTradeResults(int nResults){
		setTradeResults(String.valueOf(nResults));
	}
	public void setTradeResults(String sResults){
		jTradeResult.setText(sResults);
	}
	public void setPriceResults(int nResults){
		setPriceResults(String.valueOf(nResults));
	}
	public void setPriceResults(String sResults){
		jPriceResult.setText(sResults);
	}
	public void setTradeSearchProgress(int nProgress){
		if (nProgress >= 100){
			jTradeTimer.start();
		} else {
			if (!jTradeSearchPanel.isVisible()) jTradeSearchPanel.setVisible(true);
			if (nProgress < 0) nProgress = 0;
			jTradeSearch.setValue(nProgress);
		}
	}
	public void setPriceSearchProgress(int nProgress){
		if (nProgress >= 100){
			jPriceTimer.start();
		} else {
			if (!jPriceSearchPanel.isVisible()) jPriceSearchPanel.setVisible(true);
			if (nProgress < 0) nProgress = 0;
			jPriceSearch.setValue(nProgress);
		}
	}
	public void setSettingsSaveProgress(int nProgress){
		if (nProgress >= 100){
			jSettingsTimer.start();
		} else {
			if (!jSettingsSavePanel.isVisible()) jSettingsSavePanel.setVisible(true);
			if (nProgress < 0) nProgress = 0;
			jSettingsSave.setValue(nProgress);
		}
	}
	private JLabel createResult(){
		JLabel jLabel = new JLabel("-");
		jLabel.setMinimumSize( new Dimension(50, STATUS_HEIGHT) );
		jLabel.setMaximumSize( new Dimension(50, STATUS_HEIGHT) );
		jLabel.setPreferredSize( new Dimension(50, STATUS_HEIGHT) );
		jLabel.setBorder( jFieldBorder );
		jLabel.setHorizontalAlignment(JLabel.CENTER);
		return jLabel;
	}
	private JProgressBar createProgress(){
		JProgressBar jProgressBar = new JProgressBar(0, 100);
		jProgressBar.setValue(100);
		return jProgressBar;
	}
	private JPanel createProgressPanel(JComponent c){
		JPanel jPanel = new JPanel();
		jPanel.setMinimumSize( new Dimension(107, STATUS_HEIGHT-2) );
		jPanel.setMaximumSize( new Dimension(107, STATUS_HEIGHT-2) );
		jPanel.setPreferredSize( new Dimension(107, STATUS_HEIGHT-2) );
		jPanel.setBorder( BorderFactory.createCompoundBorder(jFieldBorder, BorderFactory.createEmptyBorder(2,2,0,2) ) );
		jPanel.setLayout( new BoxLayout(jPanel, BoxLayout.LINE_AXIS) );
		//jPanel.setAlignmentY(JPanel.CENTER_ALIGNMENT);
		jPanel.add(c);
		jPanel.setVisible(false);
		return jPanel;
	}
	
	@Override
	public void actionPerformed(ActionEvent e) {
		if (e.getSource().equals(jTradeTimer)){
			jTradeTimer.stop();
			jTradeSearchPanel.setVisible(false);
		}
		if (e.getSource().equals(jPriceTimer)){
			jPriceTimer.stop();
			jPriceSearchPanel.setVisible(false);
		}
		if (e.getSource().equals(jSettingsTimer)){
			jSettingsTimer.stop();
			jSettingsSavePanel.setVisible(false);
		}
		
	}
}
