/*
 * SettingsView.java - This file is part of VSTrade.
 *
 * Copyright (C) 2007 Niklas Kyster Rasmussen
 *
 * VSTrade is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * VSTrade is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VSTrade; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * FILE DESCRIPTION:
 * The Settings Panel, lets you change different settings... 
 */

package vstrade.moduls.settings;

//Java
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import vstrade.language.LocaleItem;
import vstrade.shared.ModuleView;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.util.Vector;
import javax.swing.BoxLayout;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import vstrade.shared.GUIFactory;

//VSTrade
import vstrade.shared.MultiSelectionList;
import vstrade.Program;
import vstrade.io.Log;

public class SettingsView extends ModuleView implements ActionListener, MouseListener{
	private final String ACTION_BROWSE = "browseVegaStrike";
	private final String ACTION_SAVE = "SaveSettings";
	private final String ACTION_LOAD = "LoadSettings";
	
	//VegaString Path
	private JTextField jVegaStrikePath;
	private JButton jVegaStrikePathButton;
	private JPanel jVegaStrikePathTPanel;
	
	//Trade Columns
	private MultiSelectionList jTradeColumns;
	private JPanel jTradeColumnsTPanel;
	
	//Price Columns
	private MultiSelectionList jPriceColumns;
	private JPanel jPriceColumnsTPanel;
	
	//Switches
	private JPanel jSwitchTPanel;
	private JCheckBox jAutoCenter;
	private JLabel jAutoCenterLabel;
	private JCheckBox jAutoConfig;
	private JLabel jAutoConfigLabel;
	private JCheckBox jSwitchComponent;
	private JLabel jSwitchComponentLabel;
	
	//Language
	private JComboBox jLanguage;
	private JPanel jLanguageTPanel;
	
	//Save and Load
	private JButton jCancel;
	private JButton jSave;
	private JPanel jSaveTPanel;
	
	
	private VegaFileChooser fileChooser;
	
	private Program program;
	
	/** Creates a new instance of SettingsView */
	public SettingsView(Program program) {
		this.program = program;
		
		fileChooser = new VegaFileChooser(program.getSettings().getFileSettings());
		
//Vega Strike Path
		jVegaStrikePathButton = GUIFactory.jButton(this, ACTION_BROWSE);
		jVegaStrikePath = new JTextField();
		jVegaStrikePath.setPreferredSize( new Dimension(GUIFactory.FIELD_WIDTH_AND_BUTTON, GUIFactory.FIELD_HEIGTH) );
		jVegaStrikePath.setMinimumSize( new Dimension(GUIFactory.FIELD_WIDTH_AND_BUTTON, GUIFactory.FIELD_HEIGTH) );
		jVegaStrikePathTPanel = GUIFactory.jPTitle(GUIFactory.LINE_AXIS, jVegaStrikePath, jVegaStrikePathButton);
		JPanel jVegaStrikePathRPanel = GUIFactory.jPRow(jVegaStrikePathTPanel);
//Trade Columns
		jTradeColumns = GUIFactory.jMultiSelectionList();
		JScrollPane tradeScrollPanel = GUIFactory.jScrollPanel(jTradeColumns, 4);
		jTradeColumnsTPanel = GUIFactory.jPTitle(tradeScrollPanel);
//Price Columns
		jPriceColumns = GUIFactory.jMultiSelectionList();
		JScrollPane priceScrollPanel = GUIFactory.jScrollPanel(jPriceColumns, 4);
		jPriceColumnsTPanel = GUIFactory.jPTitle(priceScrollPanel);
//Columns Panel
		JPanel jColumnsRPanel = GUIFactory.jPRow(jTradeColumnsTPanel, jPriceColumnsTPanel);

//Switches
		jAutoCenterLabel = GUIFactory.jLabel(this);
		jAutoCenter = GUIFactory.jCheckBox();
		
		jAutoConfigLabel = GUIFactory.jLabel(this);
		jAutoConfig = GUIFactory.jCheckBox();
		
		jSwitchComponentLabel = GUIFactory.jLabel(this);
		jSwitchComponent = GUIFactory.jCheckBox();
		
		JPanel jSwitchCPanel = GUIFactory.jPColumn(
				GUIFactory.jPCheckBox(jAutoCenter, jAutoCenterLabel),
				GUIFactory.jPCheckBox(jAutoConfig, jAutoConfigLabel),
				GUIFactory.jPCheckBox(jSwitchComponent, jSwitchComponentLabel)
				);
		
		jSwitchTPanel = GUIFactory.jPTitle(jSwitchCPanel);
//Language
		jLanguage = GUIFactory.jComboBox();
		jLanguageTPanel = GUIFactory.jPTitle(jLanguage);
//Save and Load
		jCancel = GUIFactory.jButton(this, ACTION_LOAD);
		jSave = GUIFactory.jButton(this, ACTION_SAVE);
		JPanel jSaveBPanel = GUIFactory.jPButton(jCancel, jSave);
		jSaveTPanel = GUIFactory.jPTitle(GUIFactory.LINE_AXIS, jSaveBPanel);
		
		JPanel jSaveLanguageCPanel = GUIFactory.jPColumn(jLanguageTPanel, jSaveTPanel);

		JPanel jButtonRPanel = GUIFactory.jPRow(jSwitchTPanel, jSaveLanguageCPanel);
	
//Main Panel
		JPanel jPanel = new JPanel();
		jPanel.setLayout( new BoxLayout(jPanel, BoxLayout.PAGE_AXIS) ); 
		jPanel.add(jVegaStrikePathRPanel);
		jPanel.add(jColumnsRPanel);
		jPanel.add(jButtonRPanel);
		
		this.setLayout(new BoxLayout(this, BoxLayout.PAGE_AXIS));
		this.add(jPanel);
	}
	
	@Override
	public void setData() {
		updateLanguage();
		jLanguage.setModel( new DefaultComboBoxModel(program.getSettings().getLanguageSettings().getLocaleItemList()) );
		
		load();
	}
	
	@Override
	public void clearData() {}
	
	@Override
	public void updateLanguage() {
		sTitle = program.getSettings().getLanguageSettings().getString("Title", "settings");
	//Vega Strike Path
		GUIFactory.setBorderTitle(jVegaStrikePathTPanel, program.getSettings().getLanguageSettings().getString("VegaStrikePath", "settings"));
		jVegaStrikePath.setToolTipText(program.getSettings().getLanguageSettings().getString("VegaStrikePathTT", "settings"));
		jVegaStrikePathButton.setText(program.getSettings().getLanguageSettings().getString("ButtonBrowse", "settings"));
	//Trade Columns
		jTradeColumns.setToolTipText(program.getSettings().getLanguageSettings().getString("TradeColumnsTT", "settings"));
		GUIFactory.setBorderTitle(jTradeColumnsTPanel, program.getSettings().getLanguageSettings().getString("TradeColumns", "settings"));
		//Update Trade Column Names
		Vector<String> tradeKeys = program.getSettings().getTradeSettings().getTradeTableColumnKeys();
		Vector<ListItem> tradeColumns = new Vector<ListItem>(tradeKeys.size());
		for (int i = 0; i < tradeKeys.size(); i++){
			tradeColumns.add( new ListItem(program.getSettings().getLanguageSettings().getString(tradeKeys.get(i), "trade"), tradeKeys.get(i)));
		}
		jTradeColumns.setListData( tradeColumns );
	//Price Columns
		jPriceColumns.setToolTipText(program.getSettings().getLanguageSettings().getString("PricesColumnsTT", "settings"));
		GUIFactory.setBorderTitle(jPriceColumnsTPanel, program.getSettings().getLanguageSettings().getString("PricesColumns", "settings"));
		//Update Price Column Names
		Vector<String> priceKeys = program.getSettings().getPriceSettings().getPriceTableColumnKeys();
		Vector<ListItem> priceColumns = new Vector<ListItem>(priceKeys.size());
		for (int i = 0; i < priceKeys.size(); i++){
			priceColumns.add( new ListItem(program.getSettings().getLanguageSettings().getString(priceKeys.get(i), "price"), priceKeys.get(i)));
		}
		jPriceColumns.setListData( priceColumns );
	//Switches
		jAutoCenterLabel.setText(program.getSettings().getLanguageSettings().getString("AutoCenter", "settings"));
		jAutoCenterLabel.setToolTipText(program.getSettings().getLanguageSettings().getString("AutoCenterTT", "settings"));
		jAutoCenter.setToolTipText(program.getSettings().getLanguageSettings().getString("AutoCenterTT", "settings"));
		jAutoConfigLabel.setText(program.getSettings().getLanguageSettings().getString("AutoConfig", "settings"));
		jAutoConfigLabel.setToolTipText(program.getSettings().getLanguageSettings().getString("AutoConfigTT", "settings"));
		jAutoConfig.setToolTipText(program.getSettings().getLanguageSettings().getString("AutoConfigTT", "settings"));
		jSwitchComponentLabel.setText(program.getSettings().getLanguageSettings().getString("SwitchComponent", "settings"));
		jSwitchComponentLabel.setToolTipText(program.getSettings().getLanguageSettings().getString("SwitchComponentTT", "settings"));
		jSwitchComponent.setToolTipText(program.getSettings().getLanguageSettings().getString("SwitchComponentTT", "settings"));
		GUIFactory.setBorderTitle(jSwitchTPanel, program.getSettings().getLanguageSettings().getString("Switches", "settings"));
	//Language
		jLanguage.setToolTipText(program.getSettings().getLanguageSettings().getString("LanguageTT", "settings"));
		GUIFactory.setBorderTitle(jLanguageTPanel, program.getSettings().getLanguageSettings().getString("Language", "settings"));
	//Save and Load
		jCancel.setText(program.getSettings().getLanguageSettings().getString("CancelButton", "settings"));
		jCancel.setToolTipText(program.getSettings().getLanguageSettings().getString("CancelButtonTT", "settings"));
		jSave.setText(program.getSettings().getLanguageSettings().getString("SaveButton", "settings"));
		jSave.setToolTipText(program.getSettings().getLanguageSettings().getString("SaveButtonTT", "settings"));
		GUIFactory.setBorderTitle(jSaveTPanel, program.getSettings().getLanguageSettings().getString("SaveSettings", "settings"));
	}
	
	private String browse(){
		//String sFind = "milky_way.xml";
		String sCurrentPath = jVegaStrikePath.getText();
		File fCurrentPath = new File(sCurrentPath); //Set Current Dir
		if (fCurrentPath.exists()){
			fileChooser.setCurrentDirectory(fCurrentPath);
		} else {
			fileChooser.setCurrentDirectory( new File(System.getProperty("user.dir")) );
		}
		int bFound = fileChooser.showOpenDialog(this); //.showDialog(this, "OK"); //.showOpenDialog(this);
		if (bFound  == JFileChooser.APPROVE_OPTION){
			String path = fileChooser.getSelectedFile().getAbsolutePath();
			if (path.lastIndexOf(File.separator) < path.length()-1) path = path + File.separator;
			return path;
		} else {
			return sCurrentPath;
		}
	}
	private void load(){
		int[] selected;
		jVegaStrikePath.setText(program.getSettings().getFileSettings().getVegaStrikePath());
		jAutoCenter.setSelected(program.getSettings().getMapSettings().isAutoCenterEnabled());
		jAutoConfig.setSelected(program.getSettings().getAutoConfig());
		jSwitchComponent.setSelected( program.getSettings().getSwitchComponent() );
		jLanguage.setSelectedItem( program.getSettings().getLanguageSettings().getCurrentLocale() );
		//Select enabled Trade table columns
		Vector<String> tradeKeys = program.getSettings().getTradeSettings().getTradeTableColumnKeys();
		Vector<String> tradeTableColumns = program.getSettings().getTradeSettings().getTradeTableColumns();
		selected = new int[tradeTableColumns.size()];
		for(int a = 0; a < tradeTableColumns.size(); a++){
			for(int b = 0; b < tradeKeys.size(); b++){
				if (tradeKeys.get(b).equals(tradeTableColumns.get(a))){
					selected[a] = b;
				}
			}
		}
		jTradeColumns.setSelectedIndices(selected);
		//Select enabled Price table columns
		Vector<String> priceKeys = program.getSettings().getPriceSettings().getPriceTableColumnKeys();
		Vector<String> priceTableColumns = program.getSettings().getPriceSettings().getPriceTableColumns();
		selected = new int[priceTableColumns.size()];
		for(int a = 0; a < priceTableColumns.size(); a++){
			for(int b = 0; b < priceKeys.size(); b++){
				if (priceKeys.get(b).equals(priceTableColumns.get(a))){
					selected[a] = b;
				}
			}
		}
		jPriceColumns.setSelectedIndices(selected);
	}
	private void save(){
		Log.write("--------------------------------");
		String sVegaStrikePath = jVegaStrikePath.getText();
		if (!program.getSettings().getFileSettings().isValidPath(sVegaStrikePath) && !jVegaStrikePath.getText().equals("")){
			Log.showWarning("The path to vega strike is not a valid.\nPlease browse to or type in a valid directory.\n\nSee the readme.txt for more information");
			//sVegaStrikePath = "";
		}
		
		program.getSettings().getFileSettings().setVegaStrikePath(sVegaStrikePath);
		program.getSettings().getMapSettings().enableAutoCenter(jAutoCenter.isSelected());
		program.getSettings().setAutoConfig(jAutoConfig.isSelected());
		program.getSettings().setSwitchComponent(jSwitchComponent.isSelected());
		LocaleItem localeItem = (LocaleItem)jLanguage.getSelectedItem();
		program.getSettings().getLanguageSettings().setCurrentLocale( localeItem );

		//Enabled select Trade table columns
		Object[] arrTradeColumns = jTradeColumns.getSelectedValues();
		Vector<String> tradeColumns = new Vector<String>(arrTradeColumns.length);
		
		for (int a = 0; a < arrTradeColumns.length; a++){
			ListItem listItem = (ListItem) arrTradeColumns[a];
			tradeColumns.add(listItem.getKey());
		}
		program.getSettings().getTradeSettings().setTradeTableColumns(tradeColumns);
		
		//Enabled select Price table columns
		Object[] arrPriceColumns = jPriceColumns.getSelectedValues();
		Vector<String> priceColumns = new Vector<String>();
		for (int a = 0; a < arrPriceColumns.length; a++){
			ListItem listItem = (ListItem) arrPriceColumns[a];
			priceColumns.add(listItem.getKey());
		}
		program.getSettings().getPriceSettings().setPriceTableColumns(priceColumns);
		
		program.save();
	}
	
	@Override
	public void actionPerformed(ActionEvent e) {
		if (ACTION_BROWSE.equals(e.getActionCommand())) {
			String path = browse();
			jVegaStrikePath.setText(path);
		}
		if (ACTION_SAVE.equals(e.getActionCommand())) {
			this.save();
		}
		if (ACTION_LOAD.equals(e.getActionCommand())) {
			this.load();
		}
	}
	
	@Override
	public void mouseClicked(MouseEvent e) {
		if (e.getSource().equals(jAutoCenterLabel)){
			jAutoCenter.setSelected( !jAutoCenter.isSelected());
		}
		if (e.getSource().equals(jAutoConfigLabel)){
			jAutoConfig.setSelected( !jAutoConfig.isSelected());
		}
		if (e.getSource().equals(jSwitchComponentLabel)){
			jSwitchComponent.setSelected( !jSwitchComponent.isSelected());
		}
	}
	
	@Override
	public void mousePressed(MouseEvent e) {}
	
	@Override
	public void mouseReleased(MouseEvent e) {}
	
	@Override
	public void mouseEntered(MouseEvent e) {}
	
	@Override
	public void mouseExited(MouseEvent e) {}
}
