/*
 * PricesView.java - This file is part of VSTrade.
 *
 * Copyright (C) 2007 Niklas Kyster Rasmussen
 *
 * VSTrade is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * VSTrade is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VSTrade; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * FILE DESCRIPTION:
 * Show prices for bases and planets
 */

package vstrade.moduls.price;

//Java
import ca.odell.glazedlists.BasicEventList;
import ca.odell.glazedlists.EventList;
import ca.odell.glazedlists.SortedList;
import ca.odell.glazedlists.swing.EventTableModel;
import ca.odell.glazedlists.swing.TableComparatorChooser;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import vstrade.Program;
import vstrade.data.Node;
import vstrade.data.bases.Base;
import vstrade.data.bases.Cargo;
import vstrade.shared.MultiSelectionList;
import vstrade.shared.NodeListModel;
import vstrade.shared.ModuleView;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.Vector;
import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.border.EtchedBorder;

//VSTrade
import vstrade.shared.GUIFactory;
import vstrade.shared.IntegerKeyListener;
import vstrade.shared.ComboSelectionManager;

public class PriceView extends ModuleView implements ActionListener, MouseListener, KeyListener {
	public final String ACTION_SEARCH = "FindPriceList";
	public final String ACTION_CLEAR = "ClearBasesTable";
	public final String ACTION_BASE_TYPE = "BaseType";
	public final String ACTION_BASE_ADD = "AddBase";
	public final String ACTION_BASE_CLEAR = "ClearBaseList";
	
	//Base
	//Buttons
	private JButton jClearList;
	//List
	private NodeListModel jBasesModel;
	private MultiSelectionList jBases;
	//Add
	private JComboBox jAddBase;
	private JComboBox jType;
	private JButton jAddButton;
	//Panel
	private JPanel pricesBasesPanel;
	//Cargo
	private JComboBox jCargo;
	private JPanel pricesCargoPanel;
	//Minimum-Maximum
	private JTextField jMinimumPrice;
	private JPanel jMinimumPriceTPanel;
	private JTextField jMaximumPrice;
	private JPanel jMaximumPriceTPanel;
	private JTextField jQuantity;
	private JPanel jQuantityTPanel;
	private JButton jClear;
	private JButton jSearch;
	private JPanel jSearchTPanel;
	//Table
	private JTable jTable;
	private PriceTableFormat priceTableFormat;
	private EventTableModel<PriceTableRow> priceTableModel;
	private EventList<PriceTableRow> priceEventList;
	//Table PopupMenu
	PriceTablePopupMenu jPriceTablePopupMenu;
	
	//Listeners
	private IntegerKeyListener textListener;
	
	//Data
	//private Bases bases;
	private Program program;
	private String sAlt;
	
	/** Creates a new instance of Prices */
	public PriceView(Program program) {
		this.program = program;
		
		textListener = new IntegerKeyListener();
		
		jPriceTablePopupMenu = new PriceTablePopupMenu(program, this);
		
//Base
		//Buttons
		jClearList = GUIFactory.jButtonSmall(this, ACTION_BASE_CLEAR);//
		JPanel jButtonPanel = GUIFactory.jPButton(jClearList);
		
		//List
		jBasesModel = new NodeListModel();
		jBases = GUIFactory.jMultiSelectionList(jBasesModel, this, this);
		
		//Add
		jType = GUIFactory.jComboBox(this, ACTION_BASE_TYPE);
		jAddBase = GUIFactory.jComboBoxSmall(null, ACTION_BASE_ADD, this);
		jAddButton =  GUIFactory.jButtonSmall(this, ACTION_BASE_ADD);
		JPanel jAddPanel = GUIFactory.jPButton(jAddBase, jAddButton);
		
		pricesBasesPanel = GUIFactory.jPTitle(GUIFactory.PAGE_AXIS, jButtonPanel, GUIFactory.jScrollPanel(jBases, 4), jType, jAddPanel);
//Cargo
		jCargo = GUIFactory.jComboBox();
		jCargo.setKeySelectionManager( new ComboSelectionManager() );
		pricesCargoPanel = GUIFactory.jPTitle(jCargo);
		
//Min Price
		jMinimumPrice = GUIFactory.jTextField(textListener);
		jMinimumPriceTPanel = GUIFactory.jPTitle(jMinimumPrice);
		
//Max Price		
		jMaximumPrice = GUIFactory.jTextField(textListener);
		jMaximumPriceTPanel = GUIFactory.jPTitle(jMaximumPrice);
		
//Max-Min Panel
		JPanel jPriceRPanel = GUIFactory.jPRow(jMinimumPriceTPanel, jMaximumPriceTPanel);
		
//Quantity
		jQuantity = GUIFactory.jTextField(textListener);
		jQuantityTPanel = GUIFactory.jPTitle(jQuantity);
		
//Buttons
		jClear = GUIFactory.jButton(this, ACTION_CLEAR);
		jSearch = GUIFactory.jButton(this, ACTION_SEARCH);
		JPanel jSearchBPanel = GUIFactory.jPButton(jClear, jSearch);
		jSearchTPanel = GUIFactory.jPTitle(GUIFactory.LINE_AXIS, jSearchBPanel);
		
//Table
		priceEventList = new BasicEventList<PriceTableRow>();
		SortedList<PriceTableRow> priceSortedList = new SortedList<PriceTableRow>(priceEventList);
		priceTableFormat = new PriceTableFormat(program);
		priceTableModel = new EventTableModel<PriceTableRow>(priceSortedList, priceTableFormat);
		jTable = new JTable(priceTableModel); 
		jTable.addMouseListener(this);
		jTable.addMouseListener(program);
		TableComparatorChooser<PriceTableRow> priceSorter = new TableComparatorChooser<PriceTableRow>(jTable, priceSortedList, true);
		
		JScrollPane jTableSPanel = new JScrollPane(jTable);
		jTableSPanel.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEmptyBorder(5,0,0,0), BorderFactory.createEtchedBorder(EtchedBorder.LOWERED)) );		
		
		
		//JPanel quantityPanel = GUITemplates.jRowPanel(jQuantityTPanel, jSearchTPanel);
		
		//JPanel jInputCPanel = GUIFactory.jPDimension(pricesBasesPanel.getPreferredSize(), pricesCargoPanel, jMinimumPriceTPanel, jMaximumPriceTPanel, jQuantityTPanel, jSearchTPanel); //
		JPanel jInputCPanel = GUIFactory.jPColumn(pricesCargoPanel, jMinimumPriceTPanel, jMaximumPriceTPanel, jQuantityTPanel, jSearchTPanel); //
		
//Input Panel
		JPanel jInputRPanel = GUIFactory.jPRow(jInputCPanel, GUIFactory.jPDimension(jInputCPanel.getPreferredSize(), pricesBasesPanel));
		
		JPanel jConfigPanel = new JPanel();
		jConfigPanel.setLayout( new BoxLayout(jConfigPanel, BoxLayout.PAGE_AXIS) ); 
		jConfigPanel.setBorder( BorderFactory.createEmptyBorder(0,0,5,0) );
		jConfigPanel.add(jInputRPanel);
		
//Main Panel
		JSplitPane jPanel = new JSplitPane(JSplitPane.VERTICAL_SPLIT, jConfigPanel, jTableSPanel);
		jPanel.setBorder(null);
		jPanel.setOneTouchExpandable(true);
		jPanel.setEnabled(false);
		jPanel.setDividerSize(10);
		
		this.setLayout(new BoxLayout(this, BoxLayout.PAGE_AXIS));
		this.add(jPanel);
	}
	
	@Override
	public void setData() {
		updateLanguage();
		boolean bEnable = this.program.getBases().isLoaded();
		if (bEnable){
			jType.setModel( new DefaultComboBoxModel(getTypeList()));
			jAddBase.setModel( new DefaultComboBoxModel(getBaseList()));
			jCargo.setModel( new DefaultComboBoxModel(getCargoList()));
			//jBases.
		}
	//Bases
		//Buttons
		jClearList.setEnabled(bEnable);
		//List
		jBases.setEnabled(bEnable);
		//Add
		jType.setEnabled(bEnable);
		jAddBase.setEnabled(bEnable);
		jAddButton.setEnabled(bEnable);
	//Cargo
		jCargo.setEnabled(bEnable);
	//Minimum
		jMinimumPrice.setEnabled(bEnable);
		jMaximumPrice.setEnabled(bEnable);
		jQuantity.setEnabled(bEnable);
	//Search
		jClear.setEnabled(bEnable);
		jSearch.setEnabled(bEnable);
	
	}
	
	@Override
	public void clearData() {
	//Bases
		//Clear ComboBoxes
		String[] arrDefaultType = {program.getSettings().getLanguageSettings().getString("NoTypes", "price")};
		String[] arrDefaultBase = {program.getSettings().getLanguageSettings().getString("NoBases", "price")};
		String[] arrDefaultCargo = {program.getSettings().getLanguageSettings().getString("NoCargo", "price")};
		jType.setModel( new DefaultComboBoxModel(arrDefaultType));
		jAddBase.setModel( new DefaultComboBoxModel(arrDefaultBase));
		jCargo.setModel( new DefaultComboBoxModel(arrDefaultCargo));
		//Clear list
		jBasesModel.clear();
	//Table
		//Clear table results
		priceEventList.getReadWriteLock().writeLock().lock();
		priceEventList.clear();
		priceEventList.getReadWriteLock().writeLock().unlock();
		program.getStatusView().setPriceResults("-");
		
	}
	
	@Override
	public void updateLanguage() {
		sTitle = program.getSettings().getLanguageSettings().getString("Title", "price");
		sAlt = program.getSettings().getLanguageSettings().getString("All", "price");
		jPriceTablePopupMenu.updateLanguage();
	//Bases
		//Add
		jType.setToolTipText(program.getSettings().getLanguageSettings().getString("TypeTT", "price"));
		jAddBase.setToolTipText(program.getSettings().getLanguageSettings().getString("AddTT", "price"));
		jAddButton.setText(program.getSettings().getLanguageSettings().getString("AddButton", "price"));
		jAddButton.setToolTipText(program.getSettings().getLanguageSettings().getString("AddButtonTT", "price"));
		//List
		jBases.setToolTipText(program.getSettings().getLanguageSettings().getString("BasesTT", "price"));
		//Buttons
		jClearList.setText("!");//
		jClearList.setToolTipText(program.getSettings().getLanguageSettings().getString("ClearListTT", "price"));
		//Panel
		GUIFactory.setBorderTitle(pricesBasesPanel, program.getSettings().getLanguageSettings().getString("Base", "price"));
	//Cargo
		jCargo.setToolTipText(program.getSettings().getLanguageSettings().getString("CargoTT", "price"));
		GUIFactory.setBorderTitle(pricesCargoPanel, program.getSettings().getLanguageSettings().getString("Cargo", "price"));
		jMinimumPrice.setToolTipText(program.getSettings().getLanguageSettings().getString("MinPriceTT", "price"));
		GUIFactory.setBorderTitle(jMinimumPriceTPanel, program.getSettings().getLanguageSettings().getString("MinPrice", "price"));
		jMaximumPrice.setToolTipText(program.getSettings().getLanguageSettings().getString("MaxPriceTT", "price"));
		GUIFactory.setBorderTitle(jMaximumPriceTPanel, program.getSettings().getLanguageSettings().getString("MaxPrice", "price"));
		jQuantity.setToolTipText(program.getSettings().getLanguageSettings().getString("MinQuantityTT", "price"));
		GUIFactory.setBorderTitle(jQuantityTPanel, program.getSettings().getLanguageSettings().getString("MinQuantity", "price"));
		jClear.setText(program.getSettings().getLanguageSettings().getString("Clear", "price"));
		jClear.setToolTipText(program.getSettings().getLanguageSettings().getString("ClearTT", "price"));
		jSearch.setText(program.getSettings().getLanguageSettings().getString("Search", "price"));
		jSearch.setToolTipText(program.getSettings().getLanguageSettings().getString("SearchTT", "price"));
		GUIFactory.setBorderTitle(jSearchTPanel, program.getSettings().getLanguageSettings().getString("Search", "price"));
		priceTableModel.fireTableStructureChanged();
	}
	public void setSelectedCargo(Cargo cargo){
		jCargo.setSelectedItem(cargo);
	}
	public void addBase(Base base){
		jBasesModel.add(base);
		jBases.addSelection(jBasesModel.indexOf(base), true);
	}
	public Cargo getSelectedCargo(){
		int row = jTable.getSelectedRow();
		if (row < 0) return null;
		PriceTableRow priceTableRow = priceTableModel.getElementAt(row);
		Cargo cargo = priceTableRow.getCargo();
		return cargo;
	}
	public Base getSelectedBase(){
		int row = jTable.getSelectedRow();
		if (row < 0) return null;
		PriceTableRow priceTableRow = priceTableModel.getElementAt(row);
		Base base = priceTableRow.getBase();
		return base;
	}
	
	@Override
	public void actionPerformed(ActionEvent e) {
//Type
		if (ACTION_BASE_TYPE.equals(e.getActionCommand())) {
			Vector<Node> basesList;
			String sType = (String) jType.getSelectedItem();
			if (sType.equals(getAll()) ){
				basesList = getBaseList();
			} else {
				basesList = getBaseList(sType);
			}
			jAddBase.setModel( new DefaultComboBoxModel(basesList));
		}
//Add Base
		if (ACTION_BASE_ADD.equals(e.getActionCommand())) {
			if (e.getModifiers() > 0){
				addBaseToList();
			}
		}
//Clear Base List
		if (ACTION_BASE_CLEAR.equals(e.getActionCommand())) {
			jBasesModel.clear();
		}
//Clear Table
		if (ACTION_CLEAR.equals(e.getActionCommand())) {
			priceEventList.getReadWriteLock().writeLock().lock();
			priceEventList.clear();
			priceEventList.getReadWriteLock().writeLock().unlock();
			program.getStatusView().setPriceResults("-");
		}
//Search
		if (ACTION_SEARCH.equals(e.getActionCommand())) {
			this.search();
			//statusView.setStatusbarText("Price - Search");
		}
//Table Popup Menu
		if (jPriceTablePopupMenu.ACTION_PRICE_ADD_BASE.equals(e.getActionCommand())) {
			int row = jTable.getSelectedRow();
			if (row < 0) return;
			PriceTableRow priceTableRow = priceTableModel.getElementAt(row);
			Base base = priceTableRow.getBase();
			jBasesModel.add(base);
			jBases.addSelection(jBasesModel.indexOf(base), true);
		}
		if (jPriceTablePopupMenu.ACTION_PRICE_SET_CARGO.equals(e.getActionCommand())) {
			int row = jTable.getSelectedRow();
			if (row < 0) return;
			PriceTableRow priceTableRow = priceTableModel.getElementAt(row);
			Cargo cargo = priceTableRow.getCargo();
			jCargo.setSelectedItem(cargo);
		}
		
	}
	
	@Override
	public void mouseClicked(MouseEvent e) {
		//Table PopupMenu
		if (e.getSource().equals(jTable)){
			if (e.getButton() == MouseEvent.BUTTON3){
				int i = jTable.rowAtPoint(e.getPoint());
				jTable.setRowSelectionInterval(i, i);
				jPriceTablePopupMenu.show(e.getComponent(), e.getX(), e.getY());
			}
		}
		if (e.getSource().equals(jBases) && e.getButton() == MouseEvent.BUTTON3){
			jBasesModel.remove(jBases.locationToIndex(e.getPoint()));
		}
	}
	
	@Override
	public void mousePressed(MouseEvent e) {}
	
	@Override
	public void mouseReleased(MouseEvent e) {}
	
	@Override
	public void mouseEntered(MouseEvent e) {}
	
	@Override
	public void mouseExited(MouseEvent e) {}
	
	@Override
	public void keyTyped(KeyEvent e) {}
	
	@Override
	public void keyReleased(KeyEvent e) {}
	
	@Override
	public void keyPressed(KeyEvent e) {
		if (e.getSource().equals(jAddBase) && (e.getKeyCode() == KeyEvent.VK_ENTER || e.getKeyCode() == KeyEvent.VK_SPACE) ){
			addBaseToList();
		}
		if (e.getSource().equals(jBases) && e.getKeyCode() == KeyEvent.VK_DELETE){ 
			jBasesModel.remove(jBases.getAnchorSelectionIndex());
		}
	}
	private	void search(){
		Vector<Node> arrBases;
		String sMinPrice;
		int nMinPrice;
		String sMaxPrice;
		int nMaxPrice;
		String sQuantity;
		int nQuantity;
	//Initializing
		//Get Base(s)
		if (jBasesModel.isEmpty()) return;
		arrBases = jBasesModel.getSelectedNodes(jBases);
		
		//Get Cargo
		Node cargoNode = (Node) jCargo.getSelectedItem();

		sMinPrice = jMinimumPrice.getText();
		try {
			nMinPrice = Integer.valueOf(sMinPrice);
		} catch (NumberFormatException ex) {
			//msg.printDebug("NaN error (Attempt to convert String \"sMinPrice@"+sMinPrice+"\" to Integer failed)", this.getClass(), "search()", msg.WARNING);
			nMinPrice = 0;
		}
		sMaxPrice = jMaximumPrice.getText();
		try {
			nMaxPrice = Integer.valueOf(sMaxPrice);
		} catch (NumberFormatException ex) {
			//msg.printDebug("NaN error (Attempt to convert String \"sMaxPrice@"+sMaxPrice+"\" to Integer failed)", this.getClass(), "search()", msg.WARNING);
			nMaxPrice = 0;
		}
		sQuantity = jQuantity.getText();
		try {
			nQuantity = Integer.valueOf(sQuantity);
		} catch (NumberFormatException ex) {
			//msg.printDebug("NaN error (Attempt to convert String \"sQuantity@"+sQuantity+"\" to Integer failed)", this.getClass(), "search()", msg.WARNING);
			nQuantity = 0;
		}
	//Search
		PriceSearch priceSearch = new PriceSearch(program, jClear, jSearch, priceEventList, arrBases, cargoNode, nMinPrice, nMaxPrice, nQuantity);
		priceSearch.start();
	}
	private Vector<String> getTypeList(){
		Vector<String> typeVector = this.program.getBases().getTypeList();
		//if (typeVector.get(0).equals(getAll())) typeVector.add(0, getAll());
		typeVector.add(0, getAll());
		return typeVector;
	}
	private Vector<Node> getBaseList(){
		Vector<Node> baseVector = this.program.getBases().getBaseList();
		baseVector.add(0, new Base(getAll(), ""));
		return baseVector;
	}
	private Vector<Node> getBaseList(String sType){
		Vector<Node> baseVector = this.program.getBases().getBaseList(sType);
		baseVector.add(0, new Base(getAll(sType), ""));
		return baseVector;
	}
	private Vector<Node> getCargoList(){
		Vector<Node> cargoVector = this.program.getBases().getCargoList();
		if (!cargoVector.get(0).equals(new Cargo(getAll()))){
			cargoVector.add(0, new Cargo(getAll()));
		}
		return cargoVector;
	}
	private String getAll(){
		return "["+sAlt+"]";
	}
	private String getAll(String sType){
		return "["+sAlt+" "+sType+"]";
	}
	private void addBaseToList(){
		Node node;

		node = (Node) jAddBase.getSelectedItem();
		if (node.getName().contains(sAlt)){
			Vector<Node> nodes;
			if (node.getName().equals(getAll())){
				nodes = getBaseList();
			} else {
				nodes = getBaseList((String)jType.getSelectedItem());
			}
			for (int a = 0; a < nodes.size(); a++){
				node = nodes.get(a);
				jBasesModel.add(node);
				jBases.addSelection(jBasesModel.indexOf(node), true);
			}
		} else if (!jBasesModel.contains(node)){
				jBasesModel.add(node);
				jBases.addSelection(jBasesModel.indexOf(node), true);
		}
	}
}
