/*
 * PriceTableFormat.java - This file is part of VSTrade.
 *
 * Copyright (C) 2007 Niklas Kyster Rasmussen
 *
 * VSTrade is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * VSTrade is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VSTrade; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * FILE DESCRIPTION:
 * TableFormat, used by the price table (part of the Glazed Lists implementation)
 */

package vstrade.moduls.price;

import ca.odell.glazedlists.gui.AdvancedTableFormat;
import vstrade.shared.IntegerComparator;
import vstrade.shared.StringComparator;
import java.util.Comparator;
import vstrade.Program;
import vstrade.shared.FloatComparator;


public class PriceTableFormat implements AdvancedTableFormat<PriceTableRow> {
	
	private Comparator integerComparator;
	private Comparator stringComparator;
	private Comparator floatComparator;
	
	private Program program;
	
	/** Creates a new instance of PriceTableFormat */
	public PriceTableFormat(Program program) {
		this.program = program;

		integerComparator = new IntegerComparator();
		stringComparator = new StringComparator();
		floatComparator = new FloatComparator();
	}
	
	@Override
	public Class getColumnClass(int i) {
		String sColumn = program.getSettings().getPriceSettings().getPriceTableColumn(i);
		if (sColumn.equals("TableBase")) return String.class;
		if (sColumn.equals("TableCommerce")) return String.class;
		if (sColumn.equals("TableMaxQuantity")) return Integer.class;
		if (sColumn.equals("TableAvgQuantity")) return Integer.class;
		if (sColumn.equals("TableMinQuantity")) return Integer.class;
		if (sColumn.equals("TableVolume")) return Float.class;
		if (sColumn.equals("TableMaxPrice")) return Integer.class; 
		if (sColumn.equals("TableAvgPrice")) return Integer.class;
		if (sColumn.equals("TableMinPrice")) return Integer.class;
		if (sColumn.equals("TablePercentPrice")) return Integer.class;
		return Object.class;
	}
	
	@Override
	public Comparator getColumnComparator(int i) {
		String sColumn = program.getSettings().getPriceSettings().getPriceTableColumn(i);
		if (sColumn.equals("TableBase")) return stringComparator;
		if (sColumn.equals("TableCommerce")) return stringComparator;
		if (sColumn.equals("TableMaxQuantity")) return integerComparator;
		if (sColumn.equals("TableAvgQuantity")) return integerComparator;
		if (sColumn.equals("TableMinQuantity")) return integerComparator;
		if (sColumn.equals("TableVolume")) return floatComparator;
		if (sColumn.equals("TableMaxPrice")) return integerComparator; 
		if (sColumn.equals("TableAvgPrice")) return integerComparator;
		if (sColumn.equals("TableMinPrice")) return integerComparator;
		if (sColumn.equals("TablePercentPrice")) return integerComparator;
		return null;
	}
	
	@Override
	public int getColumnCount() {
		return program.getSettings().getPriceSettings().getPriceTableColumns().size();
	}
	
	@Override
	public String getColumnName(int i) {
		return program.getSettings().getLanguageSettings().getString(program.getSettings().getPriceSettings().getPriceTableColumn(i), "price");
	}
	
	@Override
	public Object getColumnValue(PriceTableRow row, int i) {
		String sColumn = program.getSettings().getPriceSettings().getPriceTableColumn(i);
		if (sColumn.equals("TableBase")) return row.getBaseName();
		if (sColumn.equals("TableCommerce")) return row.getCargoName();
		if (sColumn.equals("TableMaxQuantity")) return row.getQuantityMaximum();
		if (sColumn.equals("TableAvgQuantity")) return row.getQuantityAverage();
		if (sColumn.equals("TableMinQuantity")) return row.getQuantityMinimum();
		if (sColumn.equals("TableVolume")) return row.getCargoVolume();
		if (sColumn.equals("TableMaxPrice")) return row.getPriceMaximum(); 
		if (sColumn.equals("TableAvgPrice")) return row.getPriceAverage();
		if (sColumn.equals("TableMinPrice")) return row.getPriceMinimum();
		if (sColumn.equals("TablePercentPrice")) return row.getPricePercent();
		return new String();
	}
}
