/*
 * PriceSearch.java - This file is part of VSTrade.
 *
 * Copyright (C) 2007 Niklas Kyster Rasmussen
 *
 * VSTrade is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * VSTrade is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VSTrade; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * FILE DESCRIPTION:
 * Search for commerce that fits the min-/maximums for the search
 */

package vstrade.moduls.price;

//Java
import ca.odell.glazedlists.EventList;
import java.util.List;
import java.util.Vector;
import javax.swing.JButton;

//VSTrade
import vstrade.Program;
import vstrade.data.Node;
import vstrade.data.bases.Base;
import vstrade.data.bases.Cargo;

public class PriceSearch extends Thread {
	public final static String CARGO = "Cargo";
	public final static String BASE = "Base";
	
	private List<PriceTableRow> result;
	private Base base;
	private Node cargo;
	
	private Vector<Node> arrBases;
	private int nMinPrice;
	private int nMaxPrice;
	private int nQuantity;
	private EventList<PriceTableRow> priceEventList;
	
	private Program program;
	private JButton jClear;
	private JButton jSearch;
	
	/** Creates a new instance of PriceSearch */
	public PriceSearch(Program program, JButton jClear, JButton jSearch, EventList<PriceTableRow> priceEventList, Vector<Node> arrBases, Node cargo, int nMinPrice, int nMaxPrice, int nQuantity) {
		this.program = program;
		this.jClear = jClear;
		this.jSearch = jSearch;
		this.priceEventList = priceEventList;
		this.arrBases = arrBases;
		this.cargo = cargo;
		this.nMinPrice = nMinPrice;
		this.nMaxPrice = nMaxPrice;
		this.nQuantity = nQuantity;
	}
	
	@Override
	public void run() {
		jClear.setEnabled(false);
		jSearch.setEnabled(false);
		result = new Vector<PriceTableRow>();
		
		//Loop through bases
		for (int a = 0; a < arrBases.size(); a++){
			base = (Base) arrBases.get(a);
			Node cargoByPath = base.getByPath(cargo.getPath());
			if (cargoByPath == null){
				deepSearch(base); //All Cargo
			} else {
				deepSearch( cargoByPath ); //Special Category or Cargo
			}
			program.getStatusView().setPriceSearchProgress( calcProgress(arrBases.size(), a) );
		}
		program.getStatusView().setPriceResults(result.size());
		
		priceEventList.getReadWriteLock().writeLock().lock();
		priceEventList.clear();
		if (result.size() > 0){ //Nothing found
			priceEventList.addAll(result);
		}
		priceEventList.getReadWriteLock().writeLock().unlock();
		
		jClear.setEnabled(true);
		jSearch.setEnabled(true);
		
		program.getStatusView().setPriceSearchProgress(100);
		return;
	}
	private void deepSearch(Node node){
		if (node instanceof Cargo){ //Cargo - add to result
			compare((Cargo)node);
		} else { //Not Cargo - Continue search
			Vector<Node> nodes = node.getValues();
			for (int a = 0; a < nodes.size(); a++){
				deepSearch( nodes.get(a) ); //Continue the search
			}
		}
	}
	private void compare(Cargo cargo){
		if (
			(cargo.getPriceAverage() >= nMinPrice || nMinPrice == 0)
			&& (cargo.getPriceAverage() <= nMaxPrice || nMaxPrice == 0)
			&& (cargo.getQuantityAverage() >= nQuantity || nQuantity == 0)
			)
			addResult(cargo);
	}
	private void addResult(Cargo cargo){
		result.add( new PriceTableRow(base, cargo) );
	}
	private int calcProgress(float aMax, int a){
		a++;
		int progress = (int) ( (a / aMax) * 100);
		return progress;
	}
	
}
