/*
 * MapView.java - This file is part of VSTrade.
 *
 * Copyright (C) 2007 Niklas Kyster Rasmussen
 *
 * VSTrade is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * VSTrade is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VSTrade; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * FILE DESCRIPTION:
 * Map of the galaxy (GalaxyView). Controls to select systems and show/hide different things on the map. 
 */

package vstrade.moduls.map;

//Java
import ca.odell.glazedlists.BasicEventList;
import ca.odell.glazedlists.EventList;
import ca.odell.glazedlists.swing.AutoCompleteSupport;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.event.ComponentEvent;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentListener;
import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.JToolBar;
import javax.swing.KeyStroke;

import javax.swing.SwingConstants;
import vstrade.data.Node;
import vstrade.data.galaxy.GalaxySystem;
import vstrade.shared.DoubleKeyListener;
import vstrade.shared.KeyAction;
import vstrade.shared.ModuleView;
import vstrade.Program;
import vstrade.shared.GUIFactory;
import vstrade.shared.KeyActionTimer;
import vstrade.shared.AlignedListCellRenderer;

public class MapView extends ModuleView implements ActionListener, MouseWheelListener, ComponentListener  {
	//System
	private final String ACTION_VEGAMAP_SELECT = "VegamapSelect";
	private final String ACTION_VEGAMAP_CENTER = "VegamapCenter";
	private final String ACTION_MOVE_UP = "VegamapMoveUp";
	private final String ACTION_MOVE_LEFT = "VegamapMoveLeft";
	private final String ACTION_MOVE_DOWN = "VegamapMoveDown";
	private final String ACTION_MOVE_RIGHT = "VegamapMoveRight";
	
	//Toogle Buttons
	private final String ACTION_SHOW_NAMES = "VegamapShowNames";
	private final String ACTION_SHOW_JUMP_LINES = "VegamapShowJumpLines";
	private final String ACTION_SHOW_JUMPLESS_SYSTEMS = "VegamapShowJumplessSystems";
	private final String ACTION_SHOW_FACTIONS = "VegamapShowFactions";
	private final String ACTION_SHOW_SECTORS= "VegamapShowSectors";
	private final String ACTION_SHOW_SECTOR_BOXES = "VegamapShowSectorBoxes";
	private final String ACTION_SHOW_YAXIS = "VegamapShowYAxis";
	//Zoom
	private final String ACTION_VEGAMAP_SCALE = "VegamapScale";
	private final String ACTION_VEGAMAP_ZOOM_OUT = "VegamapZoomOut";
	private final String ACTION_VEGAMAP_ZOOM_IN = "VegamapZoomIn";
	
	private final int PATH_PANEL_WIDTH = 255;
	
	//Toggle buttons
	private JButton jNamesButton;
	private JButton jJumpsLinesButton;
	private JButton jJumplessSystemsButton;
	private JButton jFactionsButton;
	private JButton jSectorsButton;
	private JButton jSectorBoxesButton;
	private JButton jYaxisButton;
	
	//Select
	private JComboBox jSystemList;
	private final EventList<Node> priceSystemsEventList;
	private AutoCompleteSupport autoCompleteSupport;
	private JButton jCenter;
	private JButton jSelect;
	
	//Scale
	private JComboBox jScaleList;
	private JButton jZoomIn;
	private JButton jZoomOut;
	
	//Galax View (The Map)
	private JPanel jGalaxyPanel;
	private JTextArea jPath;
	private GalaxyView galaxyView;
	
	 
	
	private DoubleKeyListener doubleKeyListener;
	
	//Data
	private String sAlt;
	
	//Croos-Modul Interaction
	private Program program;
	
	/** Creates a new instance of MapView */
	public MapView(Program program) {
		this.program = program;
		doubleKeyListener = new DoubleKeyListener();
//Keymap
		//Names
		this.getInputMap(JPanel.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(KeyStroke.getKeyStroke("N"), ACTION_SHOW_NAMES);
		this.getActionMap().put(ACTION_SHOW_NAMES, new KeyAction(this, ACTION_SHOW_NAMES));
		//Jumps
		this.getInputMap(JPanel.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(KeyStroke.getKeyStroke("J"), ACTION_SHOW_JUMP_LINES);
		this.getActionMap().put(ACTION_SHOW_JUMP_LINES, new KeyAction(this, ACTION_SHOW_JUMP_LINES));
		//Jumpless Systems
		this.getInputMap(JPanel.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(KeyStroke.getKeyStroke("L"), ACTION_SHOW_JUMPLESS_SYSTEMS);
		this.getActionMap().put(ACTION_SHOW_JUMPLESS_SYSTEMS, new KeyAction(this, ACTION_SHOW_JUMPLESS_SYSTEMS));
		//Factions
		this.getInputMap(JPanel.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(KeyStroke.getKeyStroke("F"), ACTION_SHOW_FACTIONS);
		this.getActionMap().put(ACTION_SHOW_FACTIONS, new KeyAction(this, ACTION_SHOW_FACTIONS));
		//Sectors
		this.getInputMap(JPanel.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(KeyStroke.getKeyStroke("C"), ACTION_SHOW_SECTORS);
		this.getActionMap().put(ACTION_SHOW_SECTORS, new KeyAction(this, ACTION_SHOW_SECTORS));
		//Sector Boxes
		this.getInputMap(JPanel.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(KeyStroke.getKeyStroke("B"), ACTION_SHOW_SECTOR_BOXES);
		this.getActionMap().put(ACTION_SHOW_SECTOR_BOXES, new KeyAction(this, ACTION_SHOW_SECTOR_BOXES));
		//Invert Y-Axis
		this.getInputMap(JPanel.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(KeyStroke.getKeyStroke("R"), ACTION_SHOW_YAXIS);
		this.getActionMap().put(ACTION_SHOW_YAXIS, new KeyAction(this, ACTION_SHOW_YAXIS));
		//Zoom Out
		this.getInputMap(JPanel.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(KeyStroke.getKeyStroke('Z', InputEvent.SHIFT_DOWN_MASK), ACTION_VEGAMAP_ZOOM_OUT);
		this.getActionMap().put(ACTION_VEGAMAP_ZOOM_OUT, new KeyAction(this, ACTION_VEGAMAP_ZOOM_OUT));
		this.getInputMap(JPanel.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(KeyStroke.getKeyStroke("X"), ACTION_VEGAMAP_ZOOM_OUT);
		this.getActionMap().put(ACTION_VEGAMAP_ZOOM_OUT, new KeyAction(this, ACTION_VEGAMAP_ZOOM_OUT));
		//Zoom In
		this.getInputMap(JPanel.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(KeyStroke.getKeyStroke("Z"), ACTION_VEGAMAP_ZOOM_IN);
		this.getActionMap().put(ACTION_VEGAMAP_ZOOM_IN, new KeyAction(this, ACTION_VEGAMAP_ZOOM_IN));
		
		//Move Left
		new KeyActionTimer(this, ACTION_MOVE_LEFT, this, KeyEvent.VK_A);
		//Move Right
		new KeyActionTimer(this, ACTION_MOVE_RIGHT, this, KeyEvent.VK_D);
		//Move Up
		new KeyActionTimer(this, ACTION_MOVE_UP, this, KeyEvent.VK_W);
		//Move Down
		new KeyActionTimer(this, ACTION_MOVE_DOWN, this, KeyEvent.VK_S);
		
//Toogle Buttons:
		jNamesButton = GUIFactory.jButton(this, ACTION_SHOW_NAMES);
		jNamesButton.setSelected(program.getSettings().getMapSettings().showNames());
		
		jJumpsLinesButton = GUIFactory.jButton(this, ACTION_SHOW_JUMP_LINES);
		jJumpsLinesButton.setSelected(program.getSettings().getMapSettings().showJumpsLines());
		
		jJumplessSystemsButton = GUIFactory.jButton(this, ACTION_SHOW_JUMPLESS_SYSTEMS);
		jJumplessSystemsButton.setSelected(program.getSettings().getMapSettings().showJumplessSystems());
		
		jFactionsButton = GUIFactory.jButton(this, ACTION_SHOW_FACTIONS);
		jFactionsButton.setSelected(program.getSettings().getMapSettings().showFactions());
		
		jSectorsButton = GUIFactory.jButton(this, ACTION_SHOW_SECTORS);
		jSectorsButton.setSelected(program.getSettings().getMapSettings().showSectors());
		
		jSectorBoxesButton = GUIFactory.jButton(this, ACTION_SHOW_SECTOR_BOXES);
		jSectorBoxesButton.setSelected(program.getSettings().getMapSettings().showSectorBoxes());
		
		jYaxisButton = GUIFactory.jButton(this, ACTION_SHOW_YAXIS);
		jYaxisButton.setSelected(program.getSettings().getMapSettings().showYaxis());
//Toggle Toolbar		
		JToolBar jToggleToolBar = new JToolBar();
		jToggleToolBar.add(jNamesButton);
		jToggleToolBar.add(jJumpsLinesButton);
		jToggleToolBar.add(jJumplessSystemsButton);
		jToggleToolBar.add(jFactionsButton);
		jToggleToolBar.add(jSectorsButton);
		jToggleToolBar.add(jSectorBoxesButton);
		jToggleToolBar.add(jYaxisButton);
//System
		jSystemList = GUIFactory.jComboBox();
		jSystemList.setMaximumSize( new Dimension(120, GUIFactory.FIELD_HEIGTH) );
		jSystemList.setMinimumSize( new Dimension(120, GUIFactory.FIELD_HEIGTH) );
		jSystemList.setPreferredSize( new Dimension(120, GUIFactory.FIELD_HEIGTH) );
		priceSystemsEventList = new BasicEventList<Node>();
		autoCompleteSupport = AutoCompleteSupport.install(jSystemList, priceSystemsEventList);
		autoCompleteSupport.setStrict(true);
		
		//Buttons
		jCenter = GUIFactory.jButtonSmall(this, ACTION_VEGAMAP_CENTER);
		jSelect = GUIFactory.jButtonSmall(this, ACTION_VEGAMAP_SELECT);
//System Toolbar
		JToolBar jSelectToolBar = new JToolBar();
		jSelectToolBar.add(jSystemList);
		jSelectToolBar.add(jSelect);
		jSelectToolBar.add(jCenter);
		
//Zoom
		ScaleItem[] arrScales = { new ScaleItem(1), new ScaleItem(10), new ScaleItem(25), new ScaleItem(50), new ScaleItem(100), new ScaleItem(250), new ScaleItem(500), new ScaleItem(1000), new ScaleItem(1500), new ScaleItem(2000) };
		jScaleList = GUIFactory.jComboBox(arrScales, this, ACTION_VEGAMAP_SCALE);
		//jScaleList.set
		jScaleList.setEditable(true);
		jScaleList.setMaximumSize( new Dimension(70, GUIFactory.FIELD_HEIGTH) );
		jScaleList.setMinimumSize( new Dimension(70, GUIFactory.FIELD_HEIGTH) );
		jScaleList.setPreferredSize( new Dimension(70, GUIFactory.FIELD_HEIGTH) );
		jScaleList.getEditor().getEditorComponent().addKeyListener(doubleKeyListener);
		((JTextField) jScaleList.getEditor().getEditorComponent()).setHorizontalAlignment(SwingConstants.TRAILING);
		jScaleList.setRenderer(new AlignedListCellRenderer());
		jZoomIn = GUIFactory.jButtonSmall(this, ACTION_VEGAMAP_ZOOM_IN);
		jZoomOut = GUIFactory.jButtonSmall(this, ACTION_VEGAMAP_ZOOM_OUT);
//Zoom Tolbar
		JToolBar jZoomToolBar = new JToolBar();
		int nHeigth = (int) jZoomToolBar.getMinimumSize().getHeight();
		jZoomToolBar.add(jScaleList);
		jZoomToolBar.add(jZoomIn);
		jZoomToolBar.add(jZoomOut);
		
		
//Row Panel
		//JPanel jInputRPanel = GUIFactory.jPRow(jToggleToolBar, jSelectToolBar, jZoomToolBar);
		JPanel jInputRPanel = new JPanel();//GUIFactory.jPRow(jToggleToolBar, jSelectToolBar);
		jInputRPanel.setLayout(new FlowLayout(FlowLayout.LEADING));
		jInputRPanel.add(jToggleToolBar);
		jInputRPanel.add(jSelectToolBar);
		jInputRPanel.add(jZoomToolBar);
		
		
		
		jInputRPanel.setMinimumSize( new Dimension(Short.MAX_VALUE, GUIFactory.FIELD_HEIGTH*2+(nHeigth*4)+5) );
		jInputRPanel.setPreferredSize( new Dimension(Short.MAX_VALUE, GUIFactory.FIELD_HEIGTH*2+(nHeigth*4)+5) );
		jInputRPanel.setMaximumSize( new Dimension(Short.MAX_VALUE, GUIFactory.FIELD_HEIGTH*2+(nHeigth*4)+5) );

//Config Panel
		JPanel jInputPanel = new JPanel();
		jInputPanel.setLayout( new BoxLayout(jInputPanel, BoxLayout.PAGE_AXIS) ); 
		jInputPanel.setBorder( BorderFactory.createEmptyBorder(0,0,5,0) );
		jInputPanel.add(jInputRPanel);
//Galaxy Panel
		jGalaxyPanel = new JPanel();
		jGalaxyPanel.setLayout( new BoxLayout(jGalaxyPanel, BoxLayout.LINE_AXIS) ); 
		jGalaxyPanel.setPreferredSize( new Dimension(Short.MAX_VALUE, Short.MAX_VALUE) );
		jGalaxyPanel.setMaximumSize( new Dimension(Short.MAX_VALUE, Short.MAX_VALUE) );

//Path
		jPath = new JTextArea();
		jPath.setEditable(false);
		
		JScrollPane jPathSPanel = new JScrollPane(jPath);
		
//Path Panel
		JPanel jPathPanel = new JPanel();
		jPathPanel.setLayout( new BoxLayout(jPathPanel, BoxLayout.PAGE_AXIS) ); 
		jPathPanel.setBorder( BorderFactory.createEmptyBorder(0,0,5,0) );
		jPathPanel.add( jPathSPanel );
		
		
		JSplitPane jGalaxyPathSPanel = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, jGalaxyPanel, jPathPanel);
		jGalaxyPathSPanel.setBorder(null);
		jGalaxyPathSPanel.setOneTouchExpandable(true);
		jGalaxyPathSPanel.setDividerSize(10);
		jGalaxyPathSPanel.setEnabled(false);
		jGalaxyPathSPanel.addComponentListener(this);
		
//Main Panel
		JSplitPane jPanel = new JSplitPane(JSplitPane.VERTICAL_SPLIT, jInputPanel, jGalaxyPathSPanel);
		jPanel.setBorder(null);
		jPanel.setOneTouchExpandable(true);
		jPanel.setDividerSize(10);
		jPanel.setEnabled(false);
		
		this.setLayout(new BoxLayout(this, BoxLayout.PAGE_AXIS));
		this.add(jPanel);
	}
    @Override
	public void setData() {
		updateLanguage();
		boolean bEnabled = program.getGalaxy().isLoaded();
		if (bEnabled) {
			jGalaxyPanel.removeAll();
			galaxyView = new GalaxyView(program, jPath);
			galaxyView.addMouseWheelListener(this);
			jGalaxyPanel.add(galaxyView);
			priceSystemsEventList.getReadWriteLock().writeLock().lock();
			priceSystemsEventList.clear();
			priceSystemsEventList.addAll(program.getGalaxy().getSystems());
			priceSystemsEventList.getReadWriteLock().writeLock().unlock();
			//if (jSystemList.getItemCount() > 0) jSystemList.setSelectedIndex(0);
			jSystemList.setSelectedIndex(0);
			jScaleList.setSelectedIndex(0);
		}
		//Buttons
		jNamesButton.setEnabled(bEnabled);
		jJumpsLinesButton.setEnabled(bEnabled);
		jJumplessSystemsButton.setEnabled(bEnabled);
		jFactionsButton.setEnabled(bEnabled);
		jSectorsButton.setEnabled(bEnabled);
		jSectorBoxesButton.setEnabled(bEnabled);
		jYaxisButton.setEnabled(bEnabled);
		//System		
		jCenter.setEnabled(bEnabled);
		jSelect.setEnabled(bEnabled);
		jSystemList.setEnabled(bEnabled);
		//Zoom
		jScaleList.setEnabled(bEnabled);
		jZoomIn.setEnabled(bEnabled);
		jZoomOut.setEnabled(bEnabled);
	}
	
	@Override
	public void clearData() {
		jGalaxyPanel.removeAll();
		Node defaultSystem = new Node(program.getSettings().getLanguageSettings().getString("NoSystems", "map"));
		priceSystemsEventList.getReadWriteLock().writeLock().lock();
		priceSystemsEventList.clear();
		priceSystemsEventList.add(defaultSystem);
		priceSystemsEventList.getReadWriteLock().writeLock().unlock();
		jSystemList.setSelectedIndex(0);
	}
	
	@Override
	public void updateLanguage() {
		sTitle = program.getSettings().getLanguageSettings().getString("Title", "map");
		sAlt = program.getSettings().getLanguageSettings().getString("All", "map");
	//Systems	
		jSystemList.setToolTipText(program.getSettings().getLanguageSettings().getString("SelectSystemTT", "map"));
		jCenter.setText(program.getSettings().getLanguageSettings().getString("Center", "map"));
		jCenter.setToolTipText(program.getSettings().getLanguageSettings().getString("CenterTT", "map"));
		jSelect.setText(program.getSettings().getLanguageSettings().getString("Select", "map"));
		jSelect.setToolTipText(program.getSettings().getLanguageSettings().getString("SelectTT", "map"));
	//Toggle Buttons
		//Names
		jNamesButton.setText(program.getSettings().getLanguageSettings().getString("Names", "map"));
		jNamesButton.setToolTipText(program.getSettings().getLanguageSettings().getString("NamesTT", "map"));
		//Jump Lines
		jJumpsLinesButton.setToolTipText(program.getSettings().getLanguageSettings().getString("JumpLinesTT", "map"));
		jJumpsLinesButton.setText(program.getSettings().getLanguageSettings().getString("JumpLines", "map"));
		//Jumpless Systems
		jJumplessSystemsButton.setToolTipText(program.getSettings().getLanguageSettings().getString("JumplessSystemsTT", "map"));
		jJumplessSystemsButton.setText(program.getSettings().getLanguageSettings().getString("JumplessSystems", "map"));
		//Factions
		jFactionsButton.setToolTipText(program.getSettings().getLanguageSettings().getString("FactionsTT", "map"));
		jFactionsButton.setText(program.getSettings().getLanguageSettings().getString("Factions", "map"));
		//Sectors
		jSectorsButton.setToolTipText(program.getSettings().getLanguageSettings().getString("SectorsTT", "map"));
		jSectorsButton.setText(program.getSettings().getLanguageSettings().getString("Sectors", "map"));
		//Sector Boxes
		jSectorBoxesButton.setToolTipText(program.getSettings().getLanguageSettings().getString("SectorBoxesTT", "map"));
		jSectorBoxesButton.setText(program.getSettings().getLanguageSettings().getString("SectorBoxes", "map"));
		//Invert Y-Axis
		jYaxisButton.setToolTipText(program.getSettings().getLanguageSettings().getString("YAxisTT", "map"));
		jYaxisButton.setText(program.getSettings().getLanguageSettings().getString("YAxis", "map"));
	//Zoom
		jScaleList.setToolTipText(program.getSettings().getLanguageSettings().getString("ZoomTT", "map"));
		jZoomIn.setToolTipText(program.getSettings().getLanguageSettings().getString("ZoomInTT", "map"));
		jZoomIn.setText(program.getSettings().getLanguageSettings().getString("ZoomIn", "map"));
		jZoomOut.setToolTipText(program.getSettings().getLanguageSettings().getString("ZoomOutTT", "map"));
		jZoomOut.setText(program.getSettings().getLanguageSettings().getString("ZoomOut", "map"));
	//Path
		jPath.setText(program.getSettings().getLanguageSettings().getString("NothingSelected", "map"));
	}
	
	@Override
	public void actionPerformed(ActionEvent e) {
	//Scale
		if (ACTION_VEGAMAP_SCALE.equals(e.getActionCommand())) {
			Object o = jScaleList.getSelectedItem();
			if (o instanceof ScaleItem){
				ScaleItem scale =(ScaleItem) jScaleList.getSelectedItem();
				galaxyView.rescale(scale.getScale());
			}
			if (o instanceof String){
				double d = Double.valueOf((String)o) / ScaleItem.DISPLAY_FACTOR;
				galaxyView.rescale(d);
				jScaleList.setSelectedItem( new ScaleItem(d));
			}
			
		}
	//System
		if (ACTION_VEGAMAP_CENTER.equals(e.getActionCommand())) {
			GalaxySystem system = (GalaxySystem) jSystemList.getSelectedItem();
			galaxyView.recenter(system);
		}
		if (ACTION_VEGAMAP_SELECT.equals(e.getActionCommand())) {
			GalaxySystem system = (GalaxySystem) jSystemList.getSelectedItem();
			galaxyView.selectSystem(system);
			if (program.getSettings().getMapSettings().isAutoCenterEnabled()) galaxyView.recenter(system);
		}
		//If jSystemList has focus ignore key actions...
		if (!jSystemList.getEditor().getEditorComponent().isFocusOwner()){
	//Toggle Buttons
			if (ACTION_SHOW_NAMES.equals(e.getActionCommand())) {
				jNamesButton.setSelected(!jNamesButton.isSelected());
				program.getSettings().getMapSettings().showNames(jNamesButton.isSelected());
				galaxyView.repaint();
			}
			if (ACTION_SHOW_JUMP_LINES.equals(e.getActionCommand())) {
				jJumpsLinesButton.setSelected(!jJumpsLinesButton.isSelected());
				program.getSettings().getMapSettings().showJumpsLines(jJumpsLinesButton.isSelected());
				galaxyView.repaint();
			}
			if (ACTION_SHOW_JUMPLESS_SYSTEMS.equals(e.getActionCommand())) {
				jJumplessSystemsButton.setSelected(!jJumplessSystemsButton.isSelected());
				program.getSettings().getMapSettings().showJumplessSystems(jJumplessSystemsButton.isSelected());
				galaxyView.repaint();
			}
			if (ACTION_SHOW_FACTIONS.equals(e.getActionCommand())) {
				jFactionsButton.setSelected(!jFactionsButton.isSelected());
				program.getSettings().getMapSettings().showFactions(jFactionsButton.isSelected());
				galaxyView.repaint();
			}
			if (ACTION_SHOW_SECTORS.equals(e.getActionCommand())) {
				jSectorsButton.setSelected(!jSectorsButton.isSelected());
				program.getSettings().getMapSettings().showSectors(jSectorsButton.isSelected());
				galaxyView.repaint();
			}
			if (ACTION_SHOW_SECTOR_BOXES.equals(e.getActionCommand())) {
				jSectorBoxesButton.setSelected(!jSectorBoxesButton.isSelected());
				program.getSettings().getMapSettings().showSectorBoxes(jSectorBoxesButton.isSelected());
				galaxyView.repaint();
			}
			if (ACTION_SHOW_YAXIS.equals(e.getActionCommand())) {
				jYaxisButton.setSelected(!jYaxisButton.isSelected());
				program.getSettings().getMapSettings().showYaxis(jYaxisButton.isSelected());
				galaxyView.repaint();
			}
	//Move
			if (ACTION_MOVE_LEFT.equals(e.getActionCommand())) {
				galaxyView.moveLeft();
			}
			if (ACTION_MOVE_RIGHT.equals(e.getActionCommand())) {
				galaxyView.moveRight();
			}
			if (ACTION_MOVE_DOWN.equals(e.getActionCommand())) {
				galaxyView.moveDown();
			}
			if (ACTION_MOVE_UP.equals(e.getActionCommand())) {
				galaxyView.moveUp();
			}
	//Zoom
			if (ACTION_VEGAMAP_ZOOM_IN.equals(e.getActionCommand())) {
				ScaleItem scale =(ScaleItem) jScaleList.getSelectedItem();
				double nNumber = scale.getScale();
				nNumber = nNumber * 1.3;
				nNumber = Math.round(nNumber * 10.0) / 10.0;
				//if (nNumber < 0.2) nNumber = 0.2;
				if (nNumber < 1) nNumber = 1;
				jScaleList.setSelectedItem( new ScaleItem(nNumber) );
			}
			if (ACTION_VEGAMAP_ZOOM_OUT.equals(e.getActionCommand())) {
				ScaleItem scale =(ScaleItem) jScaleList.getSelectedItem();
				double nNumber = scale.getScale();
				nNumber = nNumber * 0.7;
				nNumber = Math.round(nNumber * 10.0) / 10.0;
				//if (nNumber < 0.2) nNumber = 0.2;
				if (nNumber < 1) nNumber = 1;
				jScaleList.setSelectedItem( new ScaleItem(nNumber) );
			}
		}
	}
	
	@Override
	public void mouseWheelMoved(MouseWheelEvent e) {
		int clicks = e.getWheelRotation();
		ScaleItem scale =  (ScaleItem) jScaleList.getSelectedItem();
		double nNumber = scale.getScale();
		if (clicks != 0){
			nNumber = nNumber * (1.0 - 0.3 * clicks);
			nNumber = Math.round(nNumber * 10.0) / 10.0;
			//if (nNumber < 0.2) nNumber = 0.2;
			if (nNumber < 1) nNumber = 1;
			jScaleList.setSelectedItem( new ScaleItem(nNumber));
		}
	}

	@Override
	public void componentResized(ComponentEvent e) {
		if (e.getSource() instanceof JSplitPane){
			resizePathPanel( (JSplitPane) e.getSource() );
		}
	}

	@Override
	public void componentMoved(ComponentEvent e) {
		if (e.getSource() instanceof JSplitPane){
			resizePathPanel( (JSplitPane) e.getSource() );
		}
	}

	@Override
	public void componentShown(ComponentEvent e) {
		if (e.getSource() instanceof JSplitPane){
			resizePathPanel( (JSplitPane) e.getSource() );
		}
	}

	@Override
	public void componentHidden(ComponentEvent e) {}
	
	private void resizePathPanel(JSplitPane jSplitPane){
		jSplitPane.setDividerLocation( jSplitPane.getWidth() - PATH_PANEL_WIDTH );
	}
}
