/*
 * GalaxyView.java - This file is part of VSTrade.
 *
 * Copyright (C) 2007 Niklas Kyster Rasmussen
 *
 * Original code by: weaselflink
 * Modified by: Niklas Kyster Rasmussen
 *
 * VSTrade is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * VSTrade is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VSTrade; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * FILE DESCRIPTION:
 * Paint a map of the galaxy
 */

package vstrade.moduls.map;

//Java
import java.awt.AlphaComposite;
import java.awt.Graphics2D;
import java.awt.GraphicsEnvironment;
import java.awt.Transparency;
import java.awt.geom.Line2D;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.util.Vector;
import vstrade.data.Node;
import java.awt.Color;
import java.awt.Graphics;
import java.awt.GraphicsConfiguration;
import java.awt.GraphicsDevice;
import java.awt.Point;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;
import java.awt.geom.Point2D;
import java.util.List;
import javax.swing.JPanel;

//My
import javax.swing.text.JTextComponent;
import vstrade.Program;
import vstrade.data.galaxy.GalaxySector;
import vstrade.data.galaxy.GalaxySystem;
import vstrade.data.galaxy.Point3D;


public class GalaxyView extends JPanel implements MouseListener, MouseWheelListener {
	
	private static final int STAR_RADIUS = 2;
	//private static final int SECTOR_MARGIN = 10;
	private static final int SECTOR_MARGIN = 1;
	private static final int MOVE_SPEED = 6;

	private Point2D center;
	private double scale;
	private double scaleFactor;

	private GalaxySystem start;
	private GalaxySystem end;
	private List<GalaxySystem> path;

	private PathFinder pathFinder;
	
	
	private BufferedImage systemDotWhite;
	private BufferedImage systemDotOrange;
	private BufferedImage systemDotRed;
	
	private Program program;
	private JTextComponent jPath;

	public GalaxyView(Program program, JTextComponent jPath) {
		this.program = program;
		this.jPath = jPath;

		center = null;
		
		start = null;
		end = null;
		path = null;
		
		GraphicsEnvironment ge = GraphicsEnvironment.getLocalGraphicsEnvironment();
		GraphicsDevice gs = ge.getDefaultScreenDevice();
		GraphicsConfiguration gc = gs.getDefaultConfiguration();

		// Create an image that does not support transparency
		systemDotWhite = gc.createCompatibleImage( (STAR_RADIUS * 2) + 1, (STAR_RADIUS * 2) + 1, Transparency.OPAQUE);
		Graphics graphics = systemDotWhite.getGraphics();
		graphics.setColor(Color.WHITE);
		graphics.fillOval(0, 0, STAR_RADIUS * 2, STAR_RADIUS * 2);
		graphics.drawOval(0, 0, STAR_RADIUS * 2, STAR_RADIUS * 2);
		
		systemDotOrange = gc.createCompatibleImage( (STAR_RADIUS * 2) + 1, (STAR_RADIUS * 2) + 1, Transparency.OPAQUE);
		graphics = systemDotOrange.getGraphics();
		graphics.setColor(Color.ORANGE);
		graphics.fillOval(0, 0, STAR_RADIUS * 2, STAR_RADIUS * 2);
		graphics.drawOval(0, 0, STAR_RADIUS * 2, STAR_RADIUS * 2);
		
		systemDotRed = gc.createCompatibleImage( (STAR_RADIUS * 2) + 1, (STAR_RADIUS * 2) + 1, Transparency.OPAQUE);
		graphics = systemDotRed.getGraphics();
		graphics.setColor(Color.RED);
		graphics.fillOval(0, 0, STAR_RADIUS * 2, STAR_RADIUS * 2);
		graphics.drawOval(0, 0, STAR_RADIUS * 2, STAR_RADIUS * 2);
		
		
		pathFinder = new PathFinder();
		
		addMouseListener(this);
		addMouseWheelListener(this);
		//addKeyListener(this);

		setFocusable(true);

		setBackground(Color.BLACK);
	}
	
	private void updateFactor(){
		if (getWidth() != 0 && getHeight() != 0){
			
			//Map conrdinates width and height
			double mapXX = Math.abs(program.getGalaxy().getMax().x) + Math.abs(program.getGalaxy().getMin().x);
			double mapYY = Math.abs(program.getGalaxy().getMax().y) + Math.abs(program.getGalaxy().getMin().y);
			
			//This width and height
			double viewXX = getWidth();
			double viewYY = getHeight();
			
			//Percent of map in view @ 1:1
			double w = ((viewXX / mapXX) * 100);
			double h = ((viewYY / mapYY) * 100);
			//System.out.println("w: "+w+" h: "+h+" hmm: "+hmm);
			
			//Select the lowest percent
			if (w < h){
				scaleFactor = w / 2;
			} else {
				scaleFactor = h / 2;
			}

		} else {
			scaleFactor = 100;
		}
		//System.out.println("scaleFactor: "+scaleFactor+" Scale: "+getScale());
	}
	private void updateCenter(){
		if (center != null) return;
		double mapXX = program.getGalaxy().getMax().x + program.getGalaxy().getMin().x;
		double mapYY = program.getGalaxy().getMax().y + program.getGalaxy().getMin().y;
		center = new Point2D.Double(mapXX/2, mapYY/2);
	} 
	private double getScale(){
		return scale  / 100 * scaleFactor;
	}
	
	@Override
	protected void paintComponent(Graphics graphics) {
		super.paintComponent(graphics);
		updateFactor();
		updateCenter();

		Graphics g = graphics.create();
		if (program.getSettings().getMapSettings().showJumpsLines()) {
			paintJumps(g);
		}
		if (program.getSettings().getMapSettings().showSectorBoxes()) {
			paintSectorBoxes(g);
		}
		
		paintSystems(g);
		//if (end == null && start != null){
		if (start != null){
			paintSelected(graphics);
		}
		if (path != null) {
			paintPath(g);
		}
		
	}

	private void paintJumps(Graphics graphics) {
		Graphics g = graphics.create();

		g.setColor(Color.DARK_GRAY);

		
		Vector<Node> galaxyNodes = program.getGalaxy().getValues();
		for (int a = 0; a < galaxyNodes.size(); a++ ){
			//Log.info("galexy");
			GalaxySector sector = (GalaxySector) galaxyNodes.get(a);
			Vector<Node> sectorNodes = sector.getValues();
			for (int b = 0; b < sectorNodes.size(); b++){
				
				GalaxySystem system  = (GalaxySystem) sectorNodes.get(b);
				Vector<Node> systemNodes = system.getValues();
				//Log.info("sector: "+systemNodes.size());
				for (int c = 0; c < systemNodes.size(); c++){
					GalaxySystem systemJump  = (GalaxySystem) systemNodes.get(c);
					
					paintJump(g, system, systemJump);
				}
			}
			
		}
	}

	private void paintJump(Graphics graphics, GalaxySystem start, GalaxySystem end) {
		Point3D tmp1 = start.getPos();
		Point2D loc1 = new Point2D.Double(tmp1.getX(), tmp1.getY());
		Point p1 = getViewPoint(loc1);
		
		Point3D tmp2 = end.getPos();
		Point2D loc2 = new Point2D.Double(tmp2.getX(), tmp2.getY());
		Point p2 = getViewPoint(loc2);
		
		Line2D theLine = new Line2D.Double(p2.getX(), p2.getY(), p1.getX(), p1.getY() );
		
		//Only draw line it it's inside view
		if (theLine.intersects(0, 0, this.getWidth(), this.getHeight() ) ){
			graphics.drawLine(p1.x, p1.y, p2.x, p2.y);
		}
	}

	private void paintSectorBoxes(Graphics graphics) {
		Graphics g = graphics.create();

		g.setColor(Color.ORANGE);

		//Enumeration<String> galaxyKeys = galaxy.getKeys();
		Vector<Node> galaxyNodes = program.getGalaxy().getValues();
		for (int a = 0; a < galaxyNodes.size(); a++){
			GalaxySector sector = (GalaxySector) galaxyNodes.get(a);
			Point3D min = sector.getMin();
			Point3D max = sector.getMax();

			Point2D tmpMin = new Point2D.Double(min.x, min.y);
			Point2D tmpMax = new Point2D.Double(max.x, max.y);

			Point p1 = getViewPoint(tmpMin);
			Point p2 = getViewPoint(tmpMax);

			int x = p1.x;
			int y = Math.min(p1.y, p2.y);
			int width = Math.abs(p1.x - p2.x);
			int height = Math.abs(p1.y - p2.y);

			g.drawRect(x - SECTOR_MARGIN, y - SECTOR_MARGIN, 
					width + SECTOR_MARGIN * 2, height + SECTOR_MARGIN * 2);
			g.drawString(sector.getName(), x, y - (SECTOR_MARGIN + 5));
		}
	}

	private void paintSystems(Graphics graphics) {
		Graphics g = graphics.create();

		g.setColor(Color.WHITE);

		Vector<Node> galaxyNodes = program.getGalaxy().getValues();
		for (int a = 0; a < galaxyNodes.size(); a++){
			GalaxySector sector = (GalaxySector) galaxyNodes.get(a);
			Vector<Node> sectorNodes = sector.getValues();
			for (int b = 0; b < sectorNodes.size(); b++){
				GalaxySystem system  = (GalaxySystem) sectorNodes.get(b);
				if (system.getSize() > 0 || program.getSettings().getMapSettings().showJumplessSystems()) {
					paintSystem(g, system);
				}
			}
		}
	}

	private void paintSystem(Graphics graphics, GalaxySystem system) {
		Point3D tmp = system.getPos();
		Point2D loc = new Point2D.Double(tmp.getX(), tmp.getY());
		Point p = getViewPoint(loc);
		//Graphics2D g2d = (Graphics2D)graphics;
		//Don't draw if point is outside view
		if (p.getX() <= 0 || p.getY() <= 0 || p.getX() > this.getWidth() || p.getY() > this.getHeight()) return;
		//Paint system dot
		//...in white
		if (graphics.getColor().equals( Color.WHITE)){
			graphics.drawImage(systemDotWhite, p.x - STAR_RADIUS, p.y - STAR_RADIUS, null);
		}
		//...in red
		if (graphics.getColor().equals( Color.RED)){
				graphics.drawImage(systemDotRed, p.x - STAR_RADIUS, p.y - STAR_RADIUS, null);
		}
		//...in orange
		if (graphics.getColor().equals( Color.ORANGE)){
				graphics.drawImage(systemDotOrange, p.x - STAR_RADIUS, p.y - STAR_RADIUS, null);
		}
		
		StringBuffer name = new StringBuffer();
		if (program.getSettings().getMapSettings().showSectors()) {
			name.append(system.getParent().getName());
		}
		if (program.getSettings().getMapSettings().showNames()
				|| (path != null ? path.contains(system) : false) // If system is in path
				|| (start != null && path == null ? start.equals(system) : false) // Selected System 
				||  (start != null && path == null ? start.getValues().contains(system) : false)
			){
			if (name.length() > 0){
				name.append("/");
			}
			name.append(system.getName());
		}
		if (program.getSettings().getMapSettings().showFactions()) {
			if (name.length() > 0) {
				name.append("/");
			}
			name.append(system.getFaction());
		}
		
		//Black box behind selected systems...
		if (!graphics.getColor().equals( Color.WHITE)){
			Color tempColor = graphics.getColor();
			graphics.setColor( Color.BLACK);
			//Log.info(((Graphics2D)graphics).getComposite().toString());
			Rectangle2D huhu = graphics.getFontMetrics().getStringBounds(name.toString(), graphics);
			((Graphics2D)graphics).setComposite( AlphaComposite.getInstance(AlphaComposite.SRC_OVER, 0.85f) );
			graphics.fillRect(p.x + 5, (p.y - 5) - (int)huhu.getHeight(), (int)huhu.getWidth() + 2, (int)huhu.getHeight() + 2);
			graphics.setColor(tempColor);
			((Graphics2D)graphics).setComposite( AlphaComposite.getInstance(AlphaComposite.SRC_OVER, 1f) );
			
		}
		
		graphics.drawString(name.toString(), p.x + 6, p.y - 6);
		
		
		
	}
	private void paintPath(Graphics graphics) {
		Graphics g = graphics.create();

		g.setColor(Color.RED);

		for (int i = 0; i < path.size() - 1; i++) {
			paintJump(g, path.get(i), path.get(i + 1));
		}

		for (int i = 0; i < path.size(); i++) {
			paintSystem(g, path.get(i));
		}
	}
	private void paintSelected(Graphics graphics) {
		graphics.setColor(Color.ORANGE);
		if (start != null) {
			//Paint selected systems connection jumps + systems
			graphics.setColor(Color.ORANGE);
			if (end == null) paintConnectingJumps(graphics, start);
			//Paint selected system
			graphics.setColor(Color.RED);
			paintSystem(graphics, start);
		}
		if (end != null) {
			//paint end jump
			graphics.setColor(Color.RED);
			paintSystem(graphics, end);
			/*
			graphics.setColor(Color.ORANGE);
			paintConnectingJumps(graphics, end);
			 */
		}
	}
	private void paintConnectingJumps(Graphics graphics, GalaxySystem from){
		
		Vector<Node> systemNodes = from.getValues();
		for (int i = 0; i < systemNodes.size(); i++) {
			GalaxySystem to = (GalaxySystem) systemNodes.get(i);
			paintJump(graphics, from, to);
			paintSystem(graphics, to);
		}
	}
	public void recenter(GalaxySystem system){
		recenter(system.getPos());
	}
	public void moveLeft(){
		if(program.getGalaxy().isLoaded()){
			Point2D point = this.center;
			this.center = new Point2D.Double(point.getX()-(MOVE_SPEED/getScale()), point.getY());
			repaint();
		}
	}
	public void moveRight(){
		if(program.getGalaxy().isLoaded()){
			Point2D point = this.center;
			this.center = new Point2D.Double(point.getX()+(MOVE_SPEED/getScale()), point.getY());
			repaint();
		}
	}
	public void moveUp(){
		if(program.getGalaxy().isLoaded()){
			Point2D point = this.center;
			this.center = new Point2D.Double(point.getX(), point.getY()-(MOVE_SPEED/getScale()));
			repaint();
		}
	}
	public void moveDown(){
		if(program.getGalaxy().isLoaded()){
			Point2D point = this.center;
			this.center = new Point2D.Double(point.getX(), point.getY()+(MOVE_SPEED/getScale()));
			repaint();
		}
		
	}
	private void recenter(Point3D center) {
		Point2D loc1 = new Point2D.Double(center.getX(), center.getY());
		Point p1 = getViewPoint(loc1);
		this.center = getMapPoint(p1);

		repaint();
	}
	private void recenter(Point2D center) {
		this.center = center;

		repaint();
	}

	public void rescale(double scale) {
		this.scale = scale;
		repaint();
	}
	public void zoomIn() {
		rescale(scale * (1.0 + 0.1));
	}
	public void zoomOut() {
		rescale(scale * (1.0 - 0.1));
	}
	private Point2D getMapPoint(Point p) {
		Point vc = getViewCenter();
		Point tp;
		if (program.getSettings().getMapSettings().showYaxis()) {
			tp = new Point(p.x - vc.x, p.y - vc.y);
		} else {
			tp = new Point(p.x - vc.x, - (p.y - vc.y));
		}

		Point2D result = new Point2D.Double(
				center.getX() + tp.x / getScale(), 
				center.getY() + tp.y / getScale());

		return result;
	}

	private Point getViewPoint(Point2D p) {
		Point shift = getViewCenter();

		int x = (int)Math.round((p.getX() - center.getX()) * getScale()) + shift.x;
		int y;
		if (program.getSettings().getMapSettings().showYaxis()) {
			y = (int)Math.round((p.getY() - center.getY()) * getScale()) + shift.y;
		} else {
			y = (int)Math.round(- (p.getY() - center.getY()) * getScale()) + shift.y;
		}

		return new Point(x, y);
	}

	private Point getViewCenter() {
		return new Point(getWidth() / 2, getHeight() / 2);
	}

	private GalaxySystem getSystem(Point2D p) {
		double dist = Double.MAX_VALUE;
		GalaxySystem result = null;

		Vector<Node> galaxyNodes = program.getGalaxy().getValues();
		for (int a = 0; a < galaxyNodes.size(); a++){
			GalaxySector sector = (GalaxySector) galaxyNodes.get(a);
			// only look in sector(s) that was clicked
			if (p.getX() >= sector.getMin().x && p.getY() >= sector.getMin().y && p.getX() <= sector.getMax().x && p.getY() <= sector.getMax().y) {
				
				Vector<Node> sectorNodes = sector.getValues();
				for (int b = 0; b < sectorNodes.size(); b++){
					GalaxySystem system  = (GalaxySystem) sectorNodes.get(b);
					// filter invisible systems
					if (system.getSize() > 0 || program.getSettings().getMapSettings().showJumplessSystems()) {
						Point3D pos = system.getPos();
						double tmp = pos.distance(p.getX(), p.getY(), pos.z);

						if (tmp < dist) {
							dist = tmp;
							result = system;
						}
					}
				}
			}
		}

		return result;
	}
	public void selectSystem(GalaxySystem system) {
		if (system != null) {
			if (end != null) {
				end = null;
				start = system;
				path = null;
				printConnectingSystems();
			} else {
				if (start != null && !start.equals(system)) {
					end = system;
					path = pathFinder.getPath(start, end);
					printPath();
				} else {
					start = system;
					printConnectingSystems();
				}
			}

			repaint();
		}
	}
	private void selectSystem(Point2D p) {
		GalaxySystem system = getSystem(p);
		selectSystem(system);
	}

	private void printPath() {
		String sPath;
		if (path != null) {
			sPath = program.getSettings().getLanguageSettings().getString("PathFound", "map")+"\r\n";
			sPath = sPath.replace("%1", String.valueOf(path.size()));
			//sPath = "Path found (" + (path.size() - 1) + " jumps):\r\n";
			for (int i = 0; i < path.size(); i++) {
				sPath = sPath + "> " + path.get(i).getFullName() + "\r\n";
			}
			
		} else {
			sPath = program.getSettings().getLanguageSettings().getString("PathNotFound", "map") + "\r\n";
			sPath =  sPath + "> " + start.getFullName() + "\r\n";
			sPath = sPath + program.getSettings().getLanguageSettings().getString("And", "map") + "\r\n";
			sPath =  sPath + "> " + end.getFullName();
		}
		jPath.setText(sPath);
		//[FIXME] decite is the print should still be printed in the console
		//System.out.println(sPath);
	}
	private void printConnectingSystems() {
		String sPath;
		Vector<Node> connectedSystems = start.getValues();
		if (connectedSystems.size() > 0){
			sPath = program.getSettings().getLanguageSettings().getString("ConnectedTo", "map")+"\r\n";
			sPath = sPath.replace("%1", String.valueOf(connectedSystems.size()) );
			for (int i = 0; i < connectedSystems.size(); i++){
				GalaxySystem system = (GalaxySystem) connectedSystems.get(i);
				sPath = sPath + "> "+system.getFullName()+"\r\n";
			}
		} else {
			sPath = program.getSettings().getLanguageSettings().getString("NotConnected", "map")+"\r\n";
		}
		jPath.setText(sPath);
		//[FIXME] decite is the print should still be printed in the console
		//System.out.println(sPath);
	}
	
	@Override
	public void mouseWheelMoved(MouseWheelEvent e) {
		/*
		int clicks = e.getWheelRotation();

		if (clicks != 0) {
			rescale(scale * (1.0 - 0.1 * clicks));
		}
	 **/
	}
	 
	@Override
	public void mouseClicked(MouseEvent e) {
		Point p = e.getPoint();
		int button = e.getButton();

		if (button == MouseEvent.BUTTON1) {
			Point2D mapP = getMapPoint(p);

			selectSystem(mapP);
		}
		if (button == MouseEvent.BUTTON3) {
			Point2D mapP = getMapPoint(p);

			recenter(mapP);
		}
	}
	
	@Override
	public void mouseReleased(MouseEvent e) {}
	
	@Override
	public void mousePressed(MouseEvent e) {}

	@Override
	public void mouseExited(MouseEvent e) {}

	@Override
	public void mouseEntered(MouseEvent e) {}

	
}