/*
 * CargoView.java - This file is part of VSTrade.
 *
 * Copyright (C) 2007 Niklas Kyster Rasmussen
 *
 * VSTrade is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * VSTrade is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VSTrade; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * FILE DESCRIPTION:
 * CargoView, show information and a picture of the cargo types
 */


package vstrade.moduls.cargo;

//Java
import java.io.IOException;
import vstrade.Program;
import vstrade.data.Node;
import vstrade.data.bases.Base;
import vstrade.data.bases.Cargo;
import vstrade.shared.ModuleView;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.geom.AffineTransform;
import java.awt.image.AffineTransformOp;
import java.awt.image.BufferedImage;
import java.io.File;
import javax.imageio.ImageIO;
import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.DefaultComboBoxModel;
import javax.swing.ImageIcon;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;


//VSTrade
import javax.swing.JScrollPane;
import javax.swing.JTextPane;
import vstrade.io.Log;
import vstrade.shared.ComboSelectionManager;
import vstrade.shared.GUIFactory;


public class CargoView extends ModuleView implements ActionListener {
	private final String ACTION_CARGO_SELECTED = "InfoCargo";
	
	private JComboBox jCargo;
	private JPanel jCargoTPanel;
	private JLabel jImage;
	private JPanel jImageTPanel;
	private JLabel jCargoPath;
	private JPanel jCargoPathTPanel;
	private JTextPane jDescription;
	private JPanel jDescriptionTPanel;
	
	
	//Data
	private String sImage;
	private String sStandardPrice;
	private String sDescription;
	
	private Program program;
	
	public CargoView(Program program) {
		//this.settings = settings;
		this.program = program;
		
//Cargo
		jCargo = GUIFactory.jComboBox(this, ACTION_CARGO_SELECTED);
		jCargo.setKeySelectionManager( new ComboSelectionManager() );
		jCargoTPanel = GUIFactory.jPTitle(jCargo);
		
		JPanel cargoPanel = GUIFactory.jPRow(jCargoTPanel);
		
//Image
		jImage = new JLabel();
		jImage.setPreferredSize( new Dimension( GUIFactory.FIELD_WIDTH,  175) );
		jImage.setMinimumSize( new Dimension( GUIFactory.FIELD_WIDTH, 175) );
		jImage.setMaximumSize( new Dimension( GUIFactory.FIELD_WIDTH, 175) );
		jImage.setHorizontalAlignment(JLabel.CENTER);
		jImage.setVerticalAlignment(JLabel.CENTER);
		
		jImageTPanel = GUIFactory.jPTitle(jImage); // new  JPanel();
		JPanel jImageRPanel = GUIFactory.jPRow(jImageTPanel); //new JPanel();

//Path
		jCargoPath =  new JLabel();
		jCargoPath.setMinimumSize( new Dimension( GUIFactory.FIELD_WIDTH_X2, GUIFactory.FIELD_HEIGTH) );
		jCargoPath.setPreferredSize( new Dimension( GUIFactory.FIELD_WIDTH_X3,  GUIFactory.FIELD_HEIGTH) );
		jCargoPath.setMaximumSize( new Dimension( GUIFactory.FIELD_WIDTH_X3, GUIFactory.FIELD_HEIGTH) );
		jCargoPathTPanel = GUIFactory.jPTitle(jCargoPath);
		JPanel jCargoPathRPanel = GUIFactory.jPRow(jCargoPathTPanel);
		
//Description
		jDescription = new JTextPane();
		jDescription.setBorder(BorderFactory.createEmptyBorder(0,10,0,10));
		jDescription.setEditable(false);
		jDescription.setEnabled(false);
		jDescription.setDisabledTextColor(Color.BLACK);
		jDescription.setContentType("text/html");
		jDescription.setBackground(null);
		
		JScrollPane jDescriptionSPanel = new JScrollPane(jDescription);
		jDescriptionSPanel.setBorder(null);
		jDescriptionSPanel.setViewportView(jDescription);
		jDescriptionSPanel.setPreferredSize( new Dimension( GUIFactory.FIELD_WIDTH_X3,  Short.MAX_VALUE) );
		jDescriptionSPanel.setMaximumSize( new Dimension( GUIFactory.FIELD_WIDTH_X3,  Short.MAX_VALUE) );
		
		jDescriptionTPanel = GUIFactory.jPTitle(jDescriptionSPanel); //new  JPanel();
		
		JPanel jDescriptionPanel = new JPanel();
		jDescriptionPanel.setLayout( new BoxLayout(jDescriptionPanel, BoxLayout.LINE_AXIS) ); 
		jDescriptionPanel.setPreferredSize( new Dimension( Short.MAX_VALUE,  Short.MAX_VALUE) );
		jDescriptionPanel.setMaximumSize( new Dimension( Short.MAX_VALUE,  Short.MAX_VALUE) );
		jDescriptionPanel.add(jDescriptionTPanel);
//Main
		JPanel panel = new JPanel();
		panel.add(cargoPanel);
		panel.add(jImageRPanel);
		panel.add(jCargoPathRPanel);
		panel.add(jDescriptionPanel);
		panel.setLayout( new BoxLayout(panel, BoxLayout.PAGE_AXIS) ); 
		
		this.setLayout(new BoxLayout(this, BoxLayout.PAGE_AXIS));
		this.add(panel);
		
	}
    
    @Override
	public void setData() {
		updateLanguage();
		boolean bEnabled = program.getBases().isLoaded();
		if (bEnabled){
			jCargo.setModel( new DefaultComboBoxModel(program.getBases().getCargoList()));
			if (jCargo.getItemCount() > 0) jCargo.setSelectedIndex(0);
		}
		jCargo.setEnabled(bEnabled);
	}
    
    @Override
	public void clearData() {
		String[] arrCargo = {program.getSettings().getLanguageSettings().getString("NoCargo", "cargo")};
		jCargo.setModel( new DefaultComboBoxModel(arrCargo));
	}
    
    @Override
	public void updateLanguage() {
		sTitle = program.getSettings().getLanguageSettings().getString("Title", "cargo");
		sImage = program.getSettings().getLanguageSettings().getString("ImageTT", "cargo");
		sStandardPrice = program.getSettings().getLanguageSettings().getString("StandardPrice", "cargo");
		sDescription = program.getSettings().getLanguageSettings().getString("Description", "cargo");
		
		jCargo.setToolTipText(program.getSettings().getLanguageSettings().getString("CargoTT", "cargo"));
		GUIFactory.setBorderTitle(jCargoTPanel, program.getSettings().getLanguageSettings().getString("Cargo", "cargo"));
		GUIFactory.setBorderTitle(jImageTPanel, program.getSettings().getLanguageSettings().getString("Image", "cargo"));
		GUIFactory.setBorderTitle(jCargoPathTPanel, program.getSettings().getLanguageSettings().getString("CargoPath", "cargo"));
		GUIFactory.setBorderTitle(jDescriptionTPanel, program.getSettings().getLanguageSettings().getString("Information", "cargo"));
	}
    
	public void setSelectedCargo(Cargo cargo){
		jCargo.setSelectedItem(cargo);
	}
    
    @Override
	public void actionPerformed(ActionEvent e) {
		if (ACTION_CARGO_SELECTED.equals(e.getActionCommand())) {
			String sPath = "";
			String sImagePath = "";
			String sInfo = "";
			ImageIcon cargoImage = null;
			
			Base base = program.getBases().getTemplateBase();
			Node cargoNode = (Node) jCargo.getSelectedItem();
			sPath = cargoNode.getPath(); //Path
			if (cargoNode instanceof Cargo){
				//Load image
				Cargo cargo = (Cargo)cargoNode;
				sImagePath = program.getSettings().getFileSettings().getTexturesPath()+File.separator+cargo.getImagePath(); //Get Cargo Image Path
				File file = new File(sImagePath);
				try {
					BufferedImage image = ImageIO.read(file);
					//System.out.println("sImagePath: "+sImagePath+" W: "+image.getWidth()+" H: "+image.getHeight());
					if (image != null){
						if (image.getWidth() > 128){
							AffineTransform tx = new AffineTransform();
							double scaleFactor = 128.0 / image.getWidth();
							tx.scale(scaleFactor, scaleFactor);
							AffineTransformOp op = new AffineTransformOp(tx, AffineTransformOp.TYPE_BILINEAR);
							image = op.filter(image, null);
						}
						cargoImage = new ImageIcon(image, sImage);
						
					}
					
					//if (image == null) throw new IOException("image is null");
					
				} catch (IOException ex) {
					Log.info("Cargo not found: "+sImagePath+" "+ex.getMessage());
					
				}
				
				
				sInfo = "<html><b>"+sStandardPrice+": </b>"
						+cargo.getStandardPrice()
						+"<br><br><b>"
						+sDescription+":</b><br>"
						+cargo.getDescription();
			} else {
				sPath = sPath+"/";
			}
			//Set Path
			jCargoPath.setText("<html><b>"+sPath+"</b>");
		
			//Set Descrioption
			jDescription.setText(sInfo);
			jDescription.setCaretPosition(0);
			
			//Set Image
			jImage.setIcon(cargoImage);
		}
	}

	
}
