/*
 * ResourceBundleManager.java - This file is part of VSTrade.
 *
 * Copyright (C) 2007 Niklas Kyster Rasmussen
 *
 * VSTrade is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * VSTrade is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VSTrade; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * FILE DESCRIPTION:
 * Manage translations
 */

package vstrade.language;

//Java
import java.net.URL;
import java.util.Locale;
import java.util.ResourceBundle;
import java.util.Vector;
import vstrade.io.Log;

public class ResourceBundleManager {
	
	private Vector<LocaleItem> supportedLocals;
	private LocaleItem currentLocaleItem;
	private final String sResourceBundlePath = "vstrade/language/";
	
	private String[] propertyFiles = {"about", "cargo", "help", "map", "price", "settings", "status", "trade"};
	
	/** Creates a new instance of ResourceBundleManager */
	public ResourceBundleManager() {
		supportedLocals = new Vector<LocaleItem>();
		currentLocaleItem = new LocaleItem( new Locale("en", "US") );
		Locale[] locales = Locale.getAvailableLocales();
		
		//Find supported Locals (that are backed by properties files)
		for (int i=0; i<locales.length; i++) {
			boolean bExist = true;
			for (int b = 0; b < propertyFiles.length; b++){
				bExist = doBundleExist(propertyFiles[b], locales[i]);
				if (!bExist) break;
			}
			if (bExist) supportedLocals.add( new LocaleItem(locales[i]) );
		}
		supportedLocals.add( new LocaleItem ( "Keys", "Keys" ));
	}
	public String getString(String sKey, String sBundle){
		ResourceBundle rBundle = ResourceBundle.getBundle(sResourceBundlePath+sBundle, new Locale(currentLocaleItem.getLanguage(), currentLocaleItem.getCountry()));
		String s;
		if (currentLocaleItem.equals(new LocaleItem ( "Keys", "Keys" ))){
			return "["+sKey+"]";
		}
		try {
			s = rBundle.getString(sKey);
		} catch(Exception e)  {
			Log.warning("Translation of \""+sKey+"\" key is missing from \""+sBundle+"\" bundle \""+currentLocaleItem.getLanguage()+"\" ("+currentLocaleItem.getCountry()+")");
			s = "["+sKey+"]";
		}
		
		return s;
	}
	public Vector<LocaleItem> getLocaleItemList(){
		return supportedLocals;
	}
	public LocaleItem getCurrentLocale(){
		return currentLocaleItem;
	}
	public void setCurrentLocale(LocaleItem localeItem){
		if (supportedLocals.contains( localeItem )){
			currentLocaleItem = localeItem ;
			Log.info("Language: "+currentLocaleItem.getLanguage()+" ("+currentLocaleItem.getCountry()+")");
		}
	}
	private boolean doBundleExist(String sName, Locale locale){
		String sLocal = sName+"_"+locale.getLanguage()+"_"+locale.getCountry()+".properties";
		URL url = getClass().getResource(sLocal);
		return (url != null);
	}
}
