/*
 * SettingsWriter.java - This file is part of VSTrade.
 *
 * Copyright (C) 2007 Niklas Kyster Rasmussen
 *
 * VSTrade is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * VSTrade is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VSTrade; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * FILE DESCRIPTION:
 * Write the vstrade_settings.xml file, with the currect settings
 */

package vstrade.data.settings;

//Java
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Vector;
import vstrade.io.Log;

public class SettingsWriter {
	FileWriter fileWriter;
	Settings settings;
	/** Creates a new instance of SettingsWriter */
	public SettingsWriter() {
	}
	
	public void save(Settings settings){
		this.settings = settings;
		boolean bSucceed;
		bSucceed = openFile(new File(settings.getFileSettings().getSettingsPath()) );
		if (bSucceed) write();
		if (bSucceed) closeFile();
		if (bSucceed) Log.info("Settings saved (vstrade_settings.xml)");
	}

	private boolean write(){
		try{
			fileWriter.write("<?xml version=\"1.0\" encoding=\"ISO-8859-1\"?>\r\n");
			fileWriter.write("<!-- VSTrade Settings File -->\r\n");
			fileWriter.write("<settings>\r\n");
			fileWriter.write("	<vegastrikepath>"+settings.getFileSettings().getVegaStrikePath()+"</vegastrikepath>\r\n");
			fileWriter.write("	<autoconfig>"+settings.getAutoConfig()+"</autoconfig>\r\n");
			fileWriter.write("	<switchcomponent>"+settings.getSwitchComponent()+"</switchcomponent>\r\n");
			fileWriter.write("	<trade>\r\n");
			Vector<String> basesColumns = settings.getTradeSettings().getTradeTableColumns();
			for(int a = 0; a < basesColumns.size(); a++){
				fileWriter.write("		<tradecolumn>"+basesColumns.get(a)+"</tradecolumn>\r\n");
			}
			fileWriter.write("	</trade>\r\n");
			fileWriter.write("	<price>\r\n");
			Vector<String> pricesColumns = settings.getPriceSettings().getPriceTableColumns();
			for(int a = 0; a < pricesColumns.size(); a++){
				fileWriter.write("		<pricecolumn>"+pricesColumns.get(a)+"</pricecolumn>\r\n");
			}
			fileWriter.write("	</price>\r\n");
			fileWriter.write("	<map>\r\n");
			fileWriter.write("		<autocenter>"+settings.getMapSettings().isAutoCenterEnabled()+"</autocenter>\r\n");
			fileWriter.write("		<factions>"+settings.getMapSettings().showFactions()+"</factions>\r\n");
			fileWriter.write("		<jumplesssystems>"+settings.getMapSettings().showJumplessSystems()+"</jumplesssystems>\r\n");
			fileWriter.write("		<jumpslines>"+settings.getMapSettings().showJumpsLines()+"</jumpslines>\r\n");
			fileWriter.write("		<names>"+settings.getMapSettings().showNames()+"</names>\r\n");
			fileWriter.write("		<sectorboxes>"+settings.getMapSettings().showSectorBoxes()+"</sectorboxes>\r\n");
			fileWriter.write("		<sectors>"+settings.getMapSettings().showSectors()+"</sectors>\r\n");
			fileWriter.write("		<yaxis>"+settings.getMapSettings().showYaxis()+"</yaxis>\r\n");
			fileWriter.write("	</map>\r\n");
			fileWriter.write("	<locale>\r\n");
			fileWriter.write("		<language>"+settings.getLanguageSettings().getCurrentLocale().getLanguage()+"</language>\r\n");
			fileWriter.write("		<country>"+settings.getLanguageSettings().getCurrentLocale().getCountry()+"</country>\r\n");
			fileWriter.write("	</locale>\r\n");
			fileWriter.write("</settings>\r\n");
		} catch (IOException t){
			Log.error("Settings not saved (vstrade_settings.xml - WRITE FAILED)", t);
			return false;
		}
		return true;
	}

	private boolean closeFile(){
		try{
			fileWriter.close();
		} catch (IOException t){
			Log.error("Settings not saved (vstrade_settings.xml - CLOSE FAILED)", t);
			return false;
		}
		return true;
	}
	private boolean openFile(File filename){
		try{
			fileWriter = new FileWriter( filename );
		} catch (IOException t){
			Log.error("Settings not saved (vstrade_settings.xml - OPEN FAILED)", t);
			return false;
		}
		return true;
	}
	
}
