/*
 * SettingsReader.java - This file is part of VSTrade.
 *
 * Copyright (C) 2007 Niklas Kyster Rasmussen
 *
 * VSTrade is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * VSTrade is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VSTrade; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * FILE DESCRIPTION:
 * Read the vstrade_settings.xml file and set the settings accordingly
 */

package vstrade.data.settings;

//Java
import java.io.FileInputStream;
import java.io.InputStream;
import org.xml.sax.helpers.DefaultHandler;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.parsers.SAXParser;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import vstrade.io.Log;
import vstrade.language.LocaleItem;

public class SettingsReader extends DefaultHandler {
	
	
	Settings settings;
	InputStream file ;
	String sChars = "";
	
	String sLanguage = "";
	String sCountry = "";
	String sVariant = "";
	
	
	/** Creates a new instance of SettingsReader */
	public SettingsReader() {
	}
	
	public boolean load(Settings settings) {
		this.settings = settings;
		settings.getPriceSettings().clearTradeColumns();
		settings.getTradeSettings().clearPriceColumns();
		try {
			file = new FileInputStream(settings.getFileSettings().getSettingsPath());
		} catch (Throwable t) {
			Log.info("Using default settings (NOT FOUND: \""+settings.getFileSettings().getSettingsPath()+"\")");
			return false;
		}
		DefaultHandler handler = this;
		SAXParserFactory factory = SAXParserFactory.newInstance();
		try {
			SAXParser saxParser = factory.newSAXParser();
			saxParser.parse(file , handler);
		}  catch (Throwable t) {
			t.printStackTrace();
			Log.error("Using default settings (PARSER FAILED \""+settings.getFileSettings().getSettingsPath()+"\")", t);
			//return false;
		}
		Log.info("Settings loaded");
		return true;
	}
    
    @Override
	public void startElement(String namespaceURI, String lName, String qName, Attributes attrs)
	throws SAXException
	{
		String eName = lName; // element name
		if ("".equals(eName)) eName = qName; // namespaceAware = false
		
		
		sChars = "";
	}
    
    @Override
	public void endElement(String namespaceURI, String sName, String qName)
	throws SAXException
	{
		String eName = sName; // element name
		if ("".equals(eName)) eName = qName; // namespaceAware = false
		
		if ("vegastrikepath".equals(eName)) {
			settings.getFileSettings().setVegaStrikePath(sChars, false);
		}
		if ("autoconfig".equals(eName)) {
			settings.setAutoConfig( stringToBoolean(sChars) );
		}
		if ("switchcomponent".equals(eName)) {
			settings.setSwitchComponent( stringToBoolean(sChars) );
		}
		if ("tradecolumn".equals(eName)) {
			settings.getTradeSettings().addTradeTableColumn(sChars);
		}
		if ("pricecolumn".equals(eName)) {
			settings.getPriceSettings().addPriceTableColumn(sChars);
		}
		if ("autocenter".equals(eName)) {
			settings.getMapSettings().enableAutoCenter( stringToBoolean(sChars) );
		}
		if ("factions".equals(eName)) {
			settings.getMapSettings().showFactions( stringToBoolean(sChars) );
		}
		if ("jumplesssystems".equals(eName)) {
			settings.getMapSettings().showJumplessSystems( stringToBoolean(sChars) );
		}
		if ("jumpslines".equals(eName)) {
			settings.getMapSettings().showJumpsLines( stringToBoolean(sChars) );
		}
		if ("names".equals(eName)) {
			settings.getMapSettings().showNames( stringToBoolean(sChars) );
		}
		if ("sectorboxes".equals(eName)) {
			settings.getMapSettings().showSectorBoxes( stringToBoolean(sChars) );
		}
		if ("sectors".equals(eName)) {
			settings.getMapSettings().showSectors( stringToBoolean(sChars) );
		}
		if ("yaxis".equals(eName)) {
			settings.getMapSettings().showYaxis( stringToBoolean(sChars) );
		}
		if ("yaxis".equals(eName)) {
			settings.getMapSettings().showYaxis( stringToBoolean(sChars) );
		}
		if ("language".equals(eName)) {
			sLanguage = sChars;
		}
		if ("country".equals(eName)) {
			sCountry = sChars;
		}
		if ("locale".equals(eName)) {
			settings.getLanguageSettings().setCurrentLocale( new LocaleItem(sLanguage, sCountry));
		}
	}

    @Override
	public void characters(char buf[], int offset, int len)
	throws SAXException
	{
		String sTemp = new String(buf, offset, len);
		sChars = sChars+sTemp;
	}
	private boolean stringToBoolean(String s){
		return s.equals("true");		
	}
}
