/*
 * Paths.java - This file is part of VSTrade.
 *
 * Copyright (C) 2007 Niklas Kyster Rasmussen
 *
 * VSTrade is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * VSTrade is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VSTrade; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * FILE DESCRIPTION:
 * Hold paths to data files. Inside vegastrike directory and globaly
 */

package vstrade.data.settings;

//Java
import java.io.File;
import java.util.Hashtable;
import java.util.Vector;

public class FileSettings {
	
	public static final String MASTER_PART_LIST = "masterpartlist";
	public static final String UNITS = "units";
	public static final String UNIVERSE = "universe";
	public static final String TEXTURES = "textures";
	public static final String MAP = "map";
	public static final String AUTO = "auto";
	
	private final String sSettingsPath = (String) System.getProperty("user.dir")+java.io.File.separator+"vstrade_settings.xml";
	private final String sReadmePath = (String) System.getProperty("user.dir")+java.io.File.separator+"vstrade_readme.txt";
	private final String sFilesPath = (String) System.getProperty("user.dir")+java.io.File.separator+"vstrade_files.xml";
	private String sVegaStrikePath;
	//if VS Path is update, saving will course load, to be called
	private boolean bVSPathUpdated;
	
	private Hashtable<String, Vector<String>> keys;
	/**
	 * Creates a new instance of Paths
	 */
	public FileSettings() {
		keys = new Hashtable<String, Vector<String>>();
		
		sVegaStrikePath = "";
		bVSPathUpdated = false;
		
		FileSettingsParser pathsParser = new FileSettingsParser();
		pathsParser.load(this, sFilesPath);
	}
	public void add(String sKey, String sValue){
		if (!keys.containsKey(sKey)){
			keys.put(sKey, new Vector<String>() );
		}
		keys.get(sKey).add(sValue);
	}
	public String getVegaStrikePath(){
		return sVegaStrikePath;
	}
	public void setVegaStrikePath(String sVegaStrikePath){
		if (!this.sVegaStrikePath.equals(sVegaStrikePath)){
			bVSPathUpdated = true;
		}
		this.sVegaStrikePath = sVegaStrikePath;
	}
	public void setVegaStrikePath(String sVegaStrikePath, boolean bVSPathUpdated){
		this.bVSPathUpdated = bVSPathUpdated;
		this.sVegaStrikePath = sVegaStrikePath;
	}
	public String getSettingsPath(){
		return sSettingsPath;
	}
	public String getReadmePath() {
		return sReadmePath;
	}
	public String getMasterPartListPath(){
		return getPath(FileSettings.MASTER_PART_LIST);
	}
	public String getUnitsPath() {
		return getPath(FileSettings.UNITS);
	}
	public String getMilkyWayPath(){
		return getPath(FileSettings.UNIVERSE);
	}
	public String getTexturesPath(){
		return getPath(FileSettings.TEXTURES);
	}
	public boolean isVegaStrikePathUpdated(){
		return bVSPathUpdated;
	}
	public void isVegaStrikePathUpdated(boolean bVSPathUpdated){
		this.bVSPathUpdated = bVSPathUpdated;
	}
	public boolean isVegaStrikePathVaild(){
		return isValidPath(sVegaStrikePath);
	}
	
	public boolean isValidPath(String sNewVegaStrikePath){
		if (sNewVegaStrikePath == null) return false;
		boolean bFound;
		//milky_way.xml
		bFound = this.isPathValid(FileSettings.UNIVERSE, sNewVegaStrikePath);
		//units.csv
		if (bFound){
			bFound = this.isPathValid(FileSettings.UNITS, sNewVegaStrikePath);
		}
		
		//master_part_list.csv
		if (bFound){
			bFound = this.isPathValid(FileSettings.MASTER_PART_LIST, sNewVegaStrikePath);
		}
		return bFound;
	}
	private boolean isPathValid(String sKey, String sNewVegaStrikePath){
		if (this.getPath(sKey, sNewVegaStrikePath).equals("")){
			return false;
		}
		return true;
	}
	public String getPath(String sKey){
		return getPath(sKey, sVegaStrikePath);
	}
	public String getPath(String sKey, String sCurrentVegaStrikePath){
		sCurrentVegaStrikePath = sCurrentVegaStrikePath.replace("/", File.separator).replace("\\", File.separator);
		if (sCurrentVegaStrikePath.lastIndexOf(File.separator) < sCurrentVegaStrikePath.length()-1) sCurrentVegaStrikePath = sCurrentVegaStrikePath + File.separator;
		Vector<String> paths = keys.get(sKey);
		File file = new File("");
		if (paths != null){
			for (int i = 0; i < paths.size(); i++){
				String sPath = sCurrentVegaStrikePath+paths.get(i);
				sPath = sPath.replace("/", File.separator).replace("\\", File.separator);
				file = new File(sPath);
				if (file.exists()){
					return file.getAbsolutePath().replace("/", File.separator).replace("\\", File.separator);
				}
			}
		}
		return "";
	}
	public Vector<String> getValidVegaStrikePaths(){
		Vector<String> paths = keys.get(FileSettings.AUTO);
		File file = new File("");
		Vector<String> arrPaths = new Vector<String>();
		if (paths != null){
			for (int i = 0; i < paths.size(); i++){
				String sPath = paths.get(i);
				sPath = sPath.replace("/", File.separator).replace("\\", File.separator);
				file = new File(sPath) ;
				if (file.exists()){
					sPath = file.getAbsolutePath().replace("/", File.separator).replace("\\", File.separator);
					if ( isValidPath(sPath) ) arrPaths.add(sPath);
				}
			}
		}
			//Log.warning("No valid Vega Strike directory found (files.xml)");
			return arrPaths;
	}
	
}
