/*
 * MilkyWayReader.java - This file is part of VSTrade.
 *
 * Copyright (C) 2007 Niklas Kyster Rasmussen
 *
 * Original code by: weaselflink
 * Modified by: Niklas Kyster Rasmussen
 *
 * VSTrade is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * VSTrade is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VSTrade; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * FILE DESCRIPTION:
 * Read the milky_way.xml. And creates the Galaxy data tree
 */

package vstrade.data.galaxy;

//Java
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.util.Vector;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import vstrade.data.Node;
import vstrade.data.bases.Base;

//VSTrade
import vstrade.data.bases.Bases;
import vstrade.io.Log;

public class MilkyWayReader {

	private Galaxy galaxy;
	private Bases bases;
	/** Creates a new instance of MalkyWayReader */
	public MilkyWayReader() {
		
	}
	public boolean load(String filename, Bases bases, Galaxy galaxy){
		this.bases = bases;
		this.galaxy = galaxy;
		FileInputStream input = null;
		try {
			input = new FileInputStream(new File(filename));
		} catch (Exception ex) {
			Log.warning("No galaxy loaded (NOT FOUND: \""+filename+"\")");
			return false;
		}
		try {
			loadGalaxy(input);
		} catch (GalaxyLoadException ex) {
			Log.error("No galaxy loaded (PARSER ERROR: \""+filename+"\"))", ex);
			return false;
		}
		resolveJumps();
		Log.info("Galaxy loaded");
		return true;
	}
	private void loadGalaxy(InputStream is) throws GalaxyLoadException {
		DocumentBuilderFactory factory = null;
		DocumentBuilder builder = null;
		Document doc = null;

		try {
			factory = DocumentBuilderFactory.newInstance();
			builder = factory.newDocumentBuilder();

			doc = builder.parse(is);
			is.close();
		} catch (Exception e) {
			throw new GalaxyLoadException(e);
		}

		Element element = doc.getDocumentElement();

		parseGalaxy(element);
	}
	
	private void parseGalaxy(Element element) throws GalaxyLoadException {
		if (!element.getNodeName().equals("galaxy")) {
			throw new GalaxyLoadException("Wrong root element name.");
		}
		
		//Planets
		NodeList planetsElement = element.getElementsByTagName("planets");
		if (planetsElement.getLength() != 1) {
			throw new GalaxyLoadException("Incorrect number of \"planets\"-nodes.");
		}
		
		NodeList planetList = element.getElementsByTagName("planet");
		for (int i = 0; i < planetList.getLength(); i++) {
			Element child = (Element)planetList.item(i);
			parsePlanet(child);
		}
		//bases.debugPrintKeys();
		
		//Systems
		NodeList systemList = element.getElementsByTagName("systems");
		if (systemList.getLength() != 1) {
			throw new GalaxyLoadException("Incorrect number of \"systems\"-nodes.");
		}

		Element systemsElement = (Element)systemList.item(0);
		NodeList sectorList = element.getElementsByTagName("sector");
		for (int i = 0; i < sectorList.getLength(); i++) {
			Element child = (Element)sectorList.item(i);
			parseSector(child);
		}
	}
	
	private void parsePlanet(Element element){
		String sTrueName = element.getAttribute("name");
		sTrueName = sTrueName.replace("_", " ");//Format
		sTrueName = sTrueName + " (Planets)";
		String sTexture = "";
		NodeList varList = element.getElementsByTagName("var");
		for (int i = 0; i < varList.getLength(); i++) {
			element = (Element) varList.item(i);
			String sVarName = element.getAttribute("name");
			if (sVarName.equals("texture")){
				sTexture = element.getAttribute("value");
				//Format
				sTexture = sTexture.substring(sTexture.indexOf("/")+1);
				sTexture = sTexture.replace("_", " ");
				sTexture = sTexture.substring(0,1).toUpperCase() + sTexture.substring(1);
				sTexture = sTexture + " (Planets)";
				break;
			}
		}
		Base base = (Base) bases.getByName(sTexture);
		
		if (base != null){
			bases.remove(base, "Planets");
			base.setName(sTrueName);
			bases.add(base);
		} else {
			//System.out.println("Debug: "+sTexture+" not found");
		}
		
		//System.out.println("sTexture: "+sTexture+" = "+sTrueName);
	}
	private void parseSector(Element element) {
		String name = element.getAttribute("name");

		if (name.equals("") || name.equals("unknown_sector")) {
			return;
		}

		NodeList systemList = element.getElementsByTagName("system");
		if (systemList.getLength() < 1) {
			return;
		}

		GalaxySector sector = new GalaxySector(name); 
		
		for (int i = 0; i < systemList.getLength(); i++) {
			Element child = (Element)systemList.item(i);

			parseSystem(sector, child);
		}

		if (sector.getSize() > 0) {
			galaxy.add(sector);
		}
	}

	private void parseSystem(GalaxySector sector, Element element) {
		String name = element.getAttribute("name");
		name = name.replace("_", " ");
		if (name.equals("")) {
			return;
		}

		String xyz = "";
		String faction = "";
		String jumps = "";

		NodeList varList = element.getElementsByTagName("var");
		for (int i = 0; i < varList.getLength(); i++) {
			Element child = (Element)varList.item(i);

			String varName = child.getAttribute("name");
			

			if (varName.equals("xyz")) {
				xyz = child.getAttribute("value");
			}
			if (varName.equals("faction")) {
				faction = child.getAttribute("value");
			}
			if (varName.equals("jumps")) {
				jumps = child.getAttribute("value");
			}
		}

		Point3D pos = parsePos(xyz);
		if (pos == null) {
			return;
		}

		if (faction.equals("")) {
			faction = "unknown";
		}

		GalaxySystem system = new GalaxySystem(name, pos, faction);
		if (jumps.length() > 0) {
			String[] parts = jumps.split(" ");
			for (int i = 0; i < parts.length; i++) {
				system.addJumpString(parts[i]);
			}
		}

		sector.add(system);
	}

	private Point3D parsePos(String pos) {
		String[] parts = pos.split(" ");

		if (parts.length != 3) {
			return null;
		}

		try {
			double x = Double.parseDouble(parts[0]);
			double y = Double.parseDouble(parts[1]);
			double z = Double.parseDouble(parts[2]);

			return new Point3D(x, y, z);
		} catch (NumberFormatException e) {
			return null;
		}
	}

	public void resolveJumps() {
		Vector<Node> galaxyNodes = galaxy.getValues();
		for (int a = 0; a < galaxyNodes.size(); a++){
			GalaxySector sector = (GalaxySector) galaxyNodes.get(a);
			Vector<Node> sectorNodes = sector.getValues();
			for (int b = 0; b < sectorNodes.size(); b++){
				GalaxySystem system  = (GalaxySystem) sectorNodes.get(b);
				
				for (String jumpString : system.getJumpStrings()) {
					String parts[] = jumpString.split("/");
					if (parts.length == 2) {
						GalaxySector targetSector = (GalaxySector) galaxy.getByName(parts[0].replace("_", " "));
						if (targetSector == null) continue;
						GalaxySystem targetSystem = (GalaxySystem) targetSector.getByName(parts[1].replace("_", " "));
						if (targetSystem != null) {
							system.add(targetSystem);
						}
					} 
				}
				system.clearJumpStrings();
			}
		}
	}
}
