/*
 * UnitsParser.java - This file is part of VSTrade.
 *
 * Copyright (C) 2007 Niklas Kyster Rasmussen
 *
 * VSTrade is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * VSTrade is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VSTrade; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * FILE DESCRIPTION:
 * Reads the units.csv. Load bases and planets
 */

package vstrade.data.bases;

import com.csvreader.CsvReader;
import java.io.FileNotFoundException;
import java.io.IOException;
import vstrade.io.Log;



public class UnitsParser {
	
	private final int PATH = 0;
	private final int PRICE_MODIFIER = 1;
	private final int PRICE_FLUCTUATION = 2;
	private final int QUANTITY = 3;
	private final int QUANTITY_FLUCTUATION = 4;
	
	private boolean bEmptyRows;
	private boolean bMissingCategories;
	private String sMissingCategories;
	
	private Bases bases = null;
	private Base baseTemplate = null;
	
	private CsvReader reader;
	
	/** Creates a new instance of UnitsParser */
	public UnitsParser() {
	}
	
	public boolean load(String filename, Base baseTemplate, Bases bases){ 
		this.baseTemplate = baseTemplate;
		this.bases = bases;
		
		//boolean bVesselDockable;
		bMissingCategories = false;
		sMissingCategories = "";
		
		try {
			reader = new CsvReader(filename);
		} catch (FileNotFoundException ex) {
			Log.warning("No bases loaded (NOT FOUND: \""+filename+"\")");
			return false;
		}
		try {
			reader.setSkipEmptyRecords(true);
			reader.readHeaders();
			reader.readRecord();
			while (reader.readRecord()) {
				if (!reader.get(0).equals("")){
					createBase(reader.get("Object_Type"), reader.get("Combat_Role"), reader.get(0), reader.get("Cargo_Import"), reader.get("Dock"));
				}
			}
		} catch (IOException ex) {
			Log.error("No bases loaded (READ FAILED: \""+filename+"\")",  ex);
			return false;
		}
		reader.close();
		Log.info("Bases loaded");
		return true;
	}
	private void createBase(String sObjectType, String sCombatRole, String sFile ,String sCargo, String sDock){
		String sBaseName = formatBaseName(sFile);
		String sBaseType = formatBaseType(sFile);
		//Including Vessels:
		if ((
			sObjectType.equals("Installation")
			|| sCombatRole.equals("BASE")
			|| (sObjectType.equals("Vessel") && !sDock.equals("") && !sCargo.equals(""))
			|| sBaseType.contains("Planet")
			)
			&& !sBaseType.contains("Template")
			&& !sBaseType.contains("Blank")
			&& !sBaseName.contains("Sun")
			&& !(sBaseName+" ").contains("star ")
			&&  !sBaseName.contains("Jump") ){
		//Excluding Vessels
		//if ( (sObjectType.equals("Installation") || sBaseType.contains("Planet") ) && !sBaseType.contains("Template") && !sBaseType.contains("Blank") && !sBaseName.contains("Sun") && !(sBaseName+" ").contains("star ") && !sBaseName.contains("Jump") ){
			Base base = baseTemplate.deepClone(); //new Base(sBaseName, bases); //Create Base
			if (sObjectType.equals("Vessel")){
				if (sBaseType.equals("Standard")){
					sBaseName = sBaseName+" (Ships)";
				} else {
					sBaseName = sBaseName+" (Ships "+sBaseType+")";
				}
				sBaseType = "Ships";
			} else {
				sBaseName = sBaseName+" ("+sBaseType+")";
			}
			
			
			base.setName(sBaseName);
			base.setType(sBaseType);
			bases.add(base);
			//System.out.println("sCargo: "+sCargo);
			
			String[] arrCargo = sCargo.split("\\}\\{"); //Split each cargo in it's own group
			arrCargo[arrCargo.length-1] = arrCargo[arrCargo.length-1].replace("}", ""); //removes } from the last entry
			arrCargo[0] = arrCargo[0].replace("{", ""); //Removes { from the first entry
			for (int a = 0; a < arrCargo.length; a++){
				String sCargoTemp = arrCargo[a];
				String[] items = new String[5];
				for (int b = 0; b < items.length ; b++){
					int nEnd = sCargoTemp.indexOf(";");
					if (nEnd < 0) nEnd = sCargoTemp.length();
					items[b] = sCargoTemp.substring(0, nEnd);
					if (nEnd < sCargoTemp.length()) nEnd++;
					sCargoTemp = sCargoTemp.substring(nEnd);
				}
				updateCategory(base, items); //Update the the category data
			}
		}
	}
	private void updateCategory(Base base, String[] items){
		String sPath = items[PATH].replace("_",  " ").replace("-", " - ").replace("Agriculteral", "Agricultural");
		Category category = (Category) base.getByPath(sPath);
		if (category == null){
			if (sPath.equals("")){
				bEmptyRows = true;
			} else {
				String s = "\""+items[PATH].replace("_",  " ").replace("-", " - ")+"\"";
				int found = sMissingCategories.indexOf(s);
				if (found < 0){
					if (bMissingCategories) sMissingCategories = sMissingCategories+", ";
					sMissingCategories = sMissingCategories+"\""+items[PATH].replace("_",  " ").replace("-", " - ")+"\"";
				}
				bMissingCategories = true;
			}
			return;
		}
		//System.out.println("	Path"+items[PATH]+" PriceModifier: "+items[PRICE_MODIFIER]+" PriceFluctuation: "+items[PRICE_FLUCTUATION]+" Quantity: "+items[QUANTITY]+" QuantityFluctuation: "+items[QUANTITY_FLUCTUATION]);
		category.setPriceModifier(items[PRICE_MODIFIER]);
		category.setPriceFluctuation(items[PRICE_FLUCTUATION]);
		category.setQuantity(items[QUANTITY]);
		category.setQuantityFluctuation(items[QUANTITY_FLUCTUATION]);
	}
	private String formatBaseType(String sBaseField){
		String sBaseType = "";
		// - Unknown function - FIXME! Maybe blank amd/or template?
		if (sBaseField.indexOf(".") >= 0){
			sBaseType = sBaseField.substring(sBaseField.indexOf(".")+1);
		}
		//Non Standard Types
		if (sBaseField.indexOf("__") >= 0){
			sBaseType = sBaseField.substring(sBaseField.indexOf("__")+2);
		}
		//Standard Bases
		if (sBaseField.indexOf("__") < 0 && sBaseField.indexOf(".") < 0){
			sBaseType = "standard";
		}
		sBaseType = sBaseType.substring(0,1).toUpperCase() + sBaseType.substring(1); //Make first letter UpperCase

		// Replace _ with space and make first leter after space Uppercase
		while (sBaseType.indexOf("_") > 0) {
			int nStart = sBaseType.indexOf("_");
			sBaseType = sBaseType.substring(0, nStart) + sBaseType.substring(nStart,nStart+2).toUpperCase().replace("_", " ") + sBaseType.substring(nStart+2);
		}
		return sBaseType;
	}
	private String formatBaseName(String sBaseField){
		String sBaseName = "";
		// - Unknown function - FIXME! Maybe blank and/or template?
		if (sBaseField.indexOf(".") >= 0){
			sBaseName = sBaseField.substring(0, sBaseField.indexOf("."));
		}
		//Non Standard Types
		if (sBaseField.indexOf("__") >= 0){
			sBaseField = sBaseField.substring(0, sBaseField.indexOf("__"));
		}
		//Standard Bases
		if (sBaseField.indexOf("__") < 0 && sBaseField.indexOf(".") < 0){
			sBaseName = sBaseField;
		}
		sBaseName = sBaseName.substring(0,1).toUpperCase() + sBaseName.substring(1); //Make first letter UpperCase
		sBaseName = sBaseName.replace("_",  " ").replace("-", " - "); //More name formating
		
		return sBaseName;
	}
	
	
}
