/*
 * MasterPartListParser.java - This file is part of VSTrade.
 *
 * Copyright (C) 2007 Niklas Kyster Rasmussen
 *
 * VSTrade is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * VSTrade is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VSTrade; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * FILE DESCRIPTION:
 * Root of the base/planet tree
 */

package vstrade.data.bases;

//Java
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import com.csvreader.CsvReader;

//VSTrade
import vstrade.data.Node;
import vstrade.io.Log;

public class MasterPartListParser {
	
	private final String CATEGORY_PATH_SEPERATER = "/";
	
	private Base base = null;
	private Category parent = null;
	private Cargo cargo = null;
	private boolean post043;
	private boolean bEmptyRows;
	
	private CsvReader reader;
	
	/** Creates a new instance of MasterPartListParser */
	public MasterPartListParser() {
		
	}
	public boolean load(String filename, Base base){ 
		this.base = base;
		
		try {
			reader = new CsvReader(filename);
		} catch (FileNotFoundException ex) {
			Log.warning("No cargo loaded (NOT FOUND: \""+filename+"\")");
			return false;
		} catch (NoClassDefFoundError e){
			if (e.toString().contains("csvreader")){
				Log.error("The JavaCSV library is missing (lib\\javacsv.jar)\nPlease see vstrade_faq.txt for more information.", e);
			} else {
				Log.error(e);
			}
		}
		try {
			reader.setSkipEmptyRecords(true);
			reader.readHeaders();
			post043 = true;
			//if (reader.getColumnCount() > )
			while (reader.readRecord())
			{
				if (!reader.get("categoryname").equals("")
						&& !reader.get("file").equals("")
						&& !reader.get("price").equals("")
						&& !reader.get("description").equals("")
						&& !reader.get("volume").equals("") ){
					if (!reader.get("file").contains("DummySentinel")){
						addCategories(reader.get("categoryname")); //Create categories and set the parent for the cargo
						addCargo(reader.get("file"), reader.get("price"), reader.get("description"), reader.get("volume")); //Create the cargo
					}
				}
			}
		} catch (IOException ex) {
			Log.error("No cargo loaded (READ FAILED: \""+filename+"\")",  ex);
			return false;
		}
		reader.close();
		Log.info("Cargo loaded");
		return true;
	}
	private void addCategories(String sPath){
		sPath = sPath.replace("_", " ").replace("-", " - "); //Removes the _ and formate the - 
		String[] arrNames = sPath.split(CATEGORY_PATH_SEPERATER);
		String[] arrPath = sPath.split(CATEGORY_PATH_SEPERATER); //Used by getByPath
		for (int a = 1; a < arrPath.length; a++){
			 //Add the previuse name to the path. Giveing the full path to the category
			// ex.: {"Raw Materials", "Metals", "Aera Alloys"} is converted to: {"Raw Materials", "Raw Materials/Metals", "Raw Materials/Metals/Aera Alloys"}
			arrPath[a] = arrPath[a-1] + CATEGORY_PATH_SEPERATER + arrPath[a];
		}
		for (int a = 0; a < arrPath.length; a++){ //for each category path
			Node found = base.getByPath(arrPath[a]); //Get the category (if it exists, else return null)
			if (found != null){ //Category already exist
				parent = (Category) found; //Setting parent to the found category
			} else { //Category doesn't exist (Adding it)
				if (a == 0){ //First level of categories, added to the base
					parent = new Category(arrNames[a], base, post043);
				} else { //Secound, third ect. level of categories, added the last parent
					parent = new Category(arrNames[a], parent, post043);
				}
			}
		}	
	}
	//Create the cargo
	private void addCargo(String sName, String sPrice, String sDescription, String sCargoSpace){
		//[FIXME] should exclude/seperate ships and upgrades
		cargo = new Cargo( sName.replace("_", " ").replace("-", " - ") , parent); //Create new Cargo Object
		if (!sPrice.equals("")){ //Set the price (If any)
			cargo.setStandardPrice( sPrice );
		}
		if (!sCargoSpace.equals("")){
			cargo.setCargoVolume(sCargoSpace);
		}		
		
		if (sDescription.length() > 0 && sDescription.indexOf("@") == 0 ){ //If image path is pressent
			cargo.setDescription( sDescription.substring(sDescription.lastIndexOf("@")+1 ) ); //Set description
			//Set image path
			
			cargo.setImagePath( sDescription.substring(sDescription.indexOf("@")+1, sDescription.lastIndexOf("@")).replace("/", File.separator) );
		} else { //If no image path is pressent
			cargo.setDescription( sDescription ); //Set description
			// Set image path - no image path pressent (use default image path)
			
			cargo.setImagePath( "cargo"+File.separator+"cargo_hud.png" );
		}
	}
	
}
