/*
 * Cargo.java - This file is part of VSTrade.
 *
 * Copyright (C) 2007 Niklas Kyster Rasmussen
 *
 * VSTrade is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * VSTrade is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VSTrade; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * FILE DESCRIPTION:
 * A cargo item
 */

package vstrade.data.bases;

//Java
import java.util.Vector;
import vstrade.io.Log;

//VSTrade
import vstrade.data.Node;


public class Cargo extends Node {
	
	private String sDescription = "";
	private String sImageURL = "";
	private float nStandardPrice = 0;
	private float nCargoVolume = 1;
	
	/** Creates a new instance of Cargo */
	public Cargo(String sName) {
		this.sName = sName; 
	}
	public Cargo(String sName, Node parent) {
		this.sName = sName; 
		this.parent = parent;
		parent.add(this);
	}
	
	public String getDescription(){
		return sDescription;
	}
	public void setDescription(String sDescription){
		this.sDescription = sDescription;
	}
	public String getImagePath(){
		return sImageURL;
	}
	public void setImagePath(String sImageURL){
		this.sImageURL = sImageURL;
	}
	public void setStandardPrice(String sStandardPrice){
		float n;
		try {
			n = Float.valueOf(sStandardPrice);
		} catch (NumberFormatException ex) {
			Log.warning(this.getClass()+".setStandardPrice() :: NaN error (Attempt to convert String \"sStandardPrice@"+sStandardPrice+"\" to Float failed)");
			return;
		}
		setStandardPrice(n);
	}
	public void setStandardPrice(float n){
		nStandardPrice = n;
	}
	public int getQuantityMaximum(){
		Category parentCategory = (Category) getParent();
		if (parentCategory == null){
			Log.error(this.getClass()+".getQuantityMaximum() :: No Parent");
			return 0;
		}
		int nQuantityMaximume = (int) (parentCategory.getQuantity() + parentCategory.getQuantityFluctuation() );
		if (nQuantityMaximume < 0) nQuantityMaximume = 0; //Quantity can't below 0 (zero)
		return nQuantityMaximume;
	}
	public int getQuantityAverage(){
		Category parentCategory = (Category) getParent();
		if (parentCategory == null){
			Log.error(this.getClass()+".getQuantityAverage() :: No Parent");
			return 0;
		}
		int nQuantityAverage = (int)parentCategory.getQuantity();
		if (nQuantityAverage < 0) nQuantityAverage = 0; //Quantity can't below 0 (zero)
		return nQuantityAverage;
	}
	public int getQuantityMinimum(){
		Category parentCategory = (Category) getParent();
		if (parentCategory == null){
			Log.error(this.getClass()+".getQuantityMinimum() :: No Parent");
			return 0;
		}
		int nQuantityMinimum = (int)(parentCategory.getQuantity() - parentCategory.getQuantityFluctuation() ); 
		if (nQuantityMinimum < 0) nQuantityMinimum = 0; //Quantity can't below 0 (zero)
		return nQuantityMinimum;
	}
	public int getPriceMaximum(){
		Category parentCategory = (Category) getParent();
		if (parentCategory == null){
			Log.error(this.getClass()+".getPriceMaximum() :: No Parent");
			return 0;
		}
		return (int)((parentCategory.getPriceModifier() * nStandardPrice) + (parentCategory.getPriceFluctuation() * nStandardPrice));
	}
	public int getPriceAverage(){
		Category parentCategory = (Category) getParent();
		if (parentCategory == null){
			Log.error(this.getClass()+".getPriceAverage() :: No Parent");
			return 0;
		}
		return (int)(parentCategory.getPriceModifier() * nStandardPrice);
	}
	public int getPriceMinimum(){
		Category parentCategory = (Category) getParent();
		if (parentCategory == null){
			Log.error(this.getClass()+".getPriceMinimum() :: No Parent");
			return 0;
		}
		return (int) ((parentCategory.getPriceModifier() * nStandardPrice) - (parentCategory.getPriceFluctuation() * nStandardPrice));
	}
	public int getPricePercent(){
		Category parentCategory = (Category) getParent();
		if (parentCategory == null){
			Log.error(this.getClass()+".getPricePercent() :: No Parent");
			return 0;
		}
		return (int) (parentCategory.getPriceModifier() * 100);
	}
	public float getStandardPrice(){
		return nStandardPrice;
	}
	public Cargo deepClone(){
		Cargo copy = new Cargo(this.getName());
		copy.setStandardPrice(this.getStandardPrice());
		copy.setDescription(this.getDescription());
		copy.setImagePath(this.getImagePath());
		copy.setCargoVolume(this.getCargoVolume());
		return copy;
	}
	public static int calcPercentProfitMax(Cargo buy, Cargo sell, boolean bUpgrade){
		float nBuy = buy.getPriceMinimum();
		float nSell = sell.getPriceMaximum();
		return calcPercent(nBuy, nSell, bUpgrade);
	}
	public static int calcPercentProfitAvg(Cargo buy, Cargo sell, boolean bUpgrade){
		float nBuy = buy.getPriceAverage();
		float nSell = sell.getPriceAverage();
		return calcPercent(nBuy, nSell, bUpgrade);
	}
	public static int calcPercentProfitMin(Cargo buy, Cargo sell, boolean bUpgrade){
		float nBuy = buy.getPriceMaximum();
		float nSell = sell.getPriceMinimum();
		return calcPercent(nBuy, nSell, bUpgrade);
	}
	public static int calcCashProfitMax(Cargo buy, Cargo sell, boolean bUpgrade){
		if (bUpgrade){
			return (sell.getPriceMaximum() / 2) - buy.getPriceMinimum();
		}
		return sell.getPriceMaximum() - buy.getPriceMinimum();
	}
	public static int calcCashProfitAvg(Cargo buy, Cargo sell, boolean bUpgrade){
		if (bUpgrade){
			return (sell.getPriceAverage() / 2) - buy.getPriceAverage();
		}
		return sell.getPriceAverage() - buy.getPriceAverage();
	}
	public static int calcCashProfitMin(Cargo buy, Cargo sell, boolean bUpgrade){
		if (bUpgrade){
			return (sell.getPriceMinimum() / 2) - buy.getPriceMaximum();
		}
		return sell.getPriceMinimum() - buy.getPriceMaximum();
	}
    @Override
	public String toString(){
		Vector<Node> vectorNode = this.getParents();
		String sSpace = "";
		for (int a = 1; a < vectorNode.size(); a++){
			sSpace = sSpace + "  ";
		}
		return sSpace+getName();
	}
	private static int calcPercent(float nBuy, float nSell, boolean bUpgrade){
		if (nBuy > 0 && nSell > 0){
			if (bUpgrade){
				int result = (int) (((nSell/2) / nBuy) * 100) - 100;
				if (result > 0) return result;
			} else {
				int result = (int) ((nSell / nBuy) * 100) - 100;
				if (result > 0) return result;
			}
		}
		return 0;
	}
	public float getCargoVolume() {
		return nCargoVolume;
	}

	public void setCargoVolume(float nCargoVolume) {
		this.nCargoVolume = nCargoVolume;
	}
	public void setCargoVolume(String nCargoVolume){
		float n;
		try {
			 n = Float.valueOf(nCargoVolume);
		} catch (NumberFormatException ex) {
			Log.warning(this.getClass()+".setCargoSpace() :: NaN error (Attempt to convert String \"sCargoSpace@"+nCargoVolume+"\" to Integer failed)");
			return;
		}
		setCargoVolume(n);
	}
}
