/*
 * Bases.java - This file is part of VSTrade.
 *
 * Copyright (C) 2007 Niklas Kyster Rasmussen
 *
 * VSTrade is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * VSTrade is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VSTrade; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * FILE DESCRIPTION:
 * Root of the base/planet tree
 */

package vstrade.data.bases;

//Java
import java.util.Collections;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

//VSTrade
import vstrade.Program;
import vstrade.data.Node;

public class Bases extends Node {
	private Vector<String> typeList;
	private Vector<Node> cargoList = null;
	private Hashtable<String, Hashtable<String, Node>> basesByTypeNodeLists;
	private Base templateBase;
	
	private MasterPartListParser masterPartListParser;
	private UnitsParser unitsParser;
	
	private boolean isLoaded;
	
	/** Creates a new instance of Bases */
	public Bases(String sName, Base templateBase) {
		this.sName = sName;
		this.templateBase = templateBase;
		typeList = new Vector<String>();
		basesByTypeNodeLists = new Hashtable<String, Hashtable<String, Node>>();
		cargoList = new Vector<Node>();
	}
	/** Creates a new instance of Bases */
	public Bases(Program program) {
		//Init
		sName = "Bases";
		typeList = new Vector<String>();
		basesByTypeNodeLists = new Hashtable<String, Hashtable<String, Node>>();
		cargoList = new Vector<Node>();
		
		templateBase = new Base("","");
		masterPartListParser = new MasterPartListParser();
		unitsParser = new UnitsParser();
		
		//Loading Cargo
		isLoaded = masterPartListParser.load(program.getSettings().getFileSettings().getMasterPartListPath(), templateBase);
		createCargoList(templateBase);
		
		//Loading Bases
		isLoaded = unitsParser.load(program.getSettings().getFileSettings().getUnitsPath(), templateBase, this);
	}
	public void add(Base n){
		super.add(n);
		if (!typeList.contains(n.getType())){
			//typeList
			
			typeList.add(n.getType());
			//basesByTypeLists
			basesByTypeNodeLists.put(n.getType(), new Hashtable<String, Node>());
		}
		basesByTypeNodeLists.get(n.getType()).put(n.getName(), n);
	}
	public Bases deepClone(){
		Bases copy = new Bases(sName, templateBase);
		Enumeration<String> keys = nodes.keys();
		while (keys.hasMoreElements()){
			Base child = ((Base) nodes.get(keys.nextElement())).deepClone();
			copy.add(child);
		}
		return copy;
	}
	public Vector<String> getTypeList(){
		Vector<String> vector = new Vector<String>(typeList);
		Collections.sort(vector);
		return vector;
	}
	
	public Vector<Node> getBaseList(){
		Vector<Node> vector = new Vector<Node>(getValues());
		Collections.sort(vector);
		return vector;
	}
	public Vector<Node> getBaseList(String key){
		Vector<Node> vector = new Vector<Node>(basesByTypeNodeLists.get(key).values());
		Collections.sort(vector);
		return vector;
	}
	public Base getTemplateBase(){
		return templateBase;
		
		//return new Base("test", "test");
	}
	public Vector<Node> getCargoList(){
		return new Vector<Node>(cargoList);
	}
	public void remove(Node n, String key){
		//[FIXME]
		//nodes and values, should be moved to Node
		//Should also remove the key argument, as it's reductant. We can always use Base.getType
		//As we can only remove Base nodes, because we only hold base notes
		
		nodes.remove(n.getName()); 
		values.remove(n);
		Hashtable<String, Node> keyHashtable = basesByTypeNodeLists.get(key);
		if (keyHashtable.size() == 1){
			typeList.remove( key );
			basesByTypeNodeLists.remove(key);
		}
		keyHashtable.remove(n.getName());
	}
	private void createCargoList(Node node){
		Vector<Node> cargovalues = node.getValues();
		for (int a = 0; a < cargovalues.size(); a++){
			Node child = cargovalues.get(a);
			if (child != null) { 
				cargoList.add(child);
				createCargoList(child);
			}
		}
	}
	public boolean isLoaded(){
		return isLoaded;
	}
	
	@Override
	public void clear(){
		super.clear();
		System.out.println("Clearing...");
		cargoList.clear();
		
		typeList.clear();
		Enumeration keys = basesByTypeNodeLists.keys();
		while (keys.hasMoreElements()){
			basesByTypeNodeLists.get(keys.nextElement()).clear();
		}
		basesByTypeNodeLists.clear();
		templateBase.clear();
		
		cargoList = null;
		typeList = null;
		basesByTypeNodeLists = null;
		templateBase = null;
	}
}
