/*
 * Node.java - This file is part of VSTrade.
 *
 * Copyright (C) 2007 Niklas Kyster Rasmussen
 *
 * VSTrade is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * VSTrade is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VSTrade; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * FILE DESCRIPTION:
 * model for all data trees
 */

package vstrade.data;

//Java
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

//VSTrade
import vstrade.data.bases.Base;
import vstrade.data.bases.Cargo;
import vstrade.io.Log;

public class Node implements Comparable<Node>{
	protected Hashtable<String, Node> nodes = null;
	protected Vector<Node> values;
	protected Node parent;
	protected String sName = "";
	/** Creates a new instance of Node
	 * @param sName     The name of the created Node
	 */					
	public Node(String sName) {
		this.sName = sName;
		nodes = new Hashtable<String, Node>();
		values = new Vector<Node>();
	}
	/** Creates a new instance of Node */
	public Node() {
		nodes = new Hashtable<String, Node>();
		values = new Vector<Node>();
	}
	/** Get the name of this Node
	 * @return          return the name of this Node
	 */
	public String getName(){
		return sName;
	}
	/** Set the name of this Node
	 * @param sName     Set the name of this Node
	 */
	public void setName(String sName){
		this.sName = sName;
	}
	/** Set the parent of this Node
	 * @param parent    Set the parent of this Node
	 */
	protected void setParent(Node parent){
		this.parent = parent;
	}
	/** Get the parent of this Node
	 * @return          return the parent of this Node.&nbsp;<code>null</code> if no parent is set. 
	 */
	public Node getParent(){
		return parent;
	}
	/** Add child Node to this node (and set this Node as Its parent)
	 * @param n         Child Node of this Node
	 */
	public void add(Node n){
		n.setParent(this);
		if (nodes.containsKey(n.getName()) ){
			Log.warning(n.getName()+" already exist in nodes");
		}
		nodes.put(n.getName(), n);
		values.add(n);
	}
	/** Get the number of child Node's
	 * @return          int with the number of child Node's
	 */
	public int getSize(){
		return nodes.size();
	}
	/** Get child nodes
	 * @return          Vector<Node> with all child Nodes.
	 */
	public Vector<Node> getValues(){
		return values;
	}
	/** Get child Node by the name of <code>s</code> argument
	 * @param s         Name of Node to get
	 * @return          Child Node by the name of <code>s</code> argument.&nbsp;return <code>null</code> if this node don't got any child by that name
	 */
	public Node getByName(String s ){
		return (Node) nodes.get(s);
	}
	/** Get Node by the path
	 * @param path      Path to the Node (from calling node)
	 * @return          The <code>Node</code> at the path.&nbsp;returns <code>null</code> if the Node doesn't exist
	 */
	public Node getByPath(String path){
		Node node = this; //Start node
		String[] arrPath = path.split("/"); //Split up the path
		int a = 0;
		if (arrPath[0].equals(this.getName())) a++;
		for (int i = a;i < arrPath.length && node != null; i++){
			node = node.getByName(arrPath[i]);
		}
		return node;
	}
	/** Get Root Note, the upper most parent
	 * @return          The root Node
	 */
	public Node getRoot(){
		Vector<Node> parents = getParents(); //Get a vector of the parents
		return parents.get(0); //Get the first node, in the vector (which is the root)
	}
	/** Get path to this Node
	 * @return          String with path to this Node
	 */
	public String getPath(){
		String sPath = this.getName(); //hold the path
		Node node = this; //Last node
		while (node.getParent() != null) { //While not root
			if (node.getParent() instanceof Base){
				return sPath;
			}
			node = node.getParent(); //Parent become current node (where we get the parent)
			if (!node.getName().equals("")) sPath = node.getName() + "/" + sPath;
		}
		return sPath;
	}
	/** Get a Vector<Node> of parents parents (until root). 
	 * @return          Vector<Node> of parnets Nodes
	 */
	public Vector<Node> getParents(){
		Vector<Node> parents = new Vector<Node>(); //hold the parents
		parents.add(this); //Add itself (Will be in the end)
		Node temp = this; //Last node
		while (temp.getParent() != null) { //While not root
			parents.add(0, temp.getParent()); //Add Parent the the start
			temp = temp.getParent(); //Parent become current node (where we get the parent)
		}
		//Root first, this node last
		return parents;
	}
	/** Check if <code>n</code> equals this Node
	 * @param n         Node to compare to
	 * @return          The value <code>true</code> if argument Node is equal to this Node. The value <code>false</code> if argument Node is not equal to this Node.
	 */
	public boolean equals(Node n){
		if (n.getName().equals(this.getName()) && n.getPath().equals(this.getPath())) return true;
		return false;
	}
	
	@Override
	/* [FIXME] need to use path as well... */
 	public int hashCode() {
 		return this.getName().hashCode();
 	}
	/** Check if <code>o</code> equals this Node
	 * @param o         Object to compare to
	 * @return          The value <code>true</code> if argument Node is equal to this Node. The value <code>false</code> if argument Node is not equal to this Node.
	 */
    @Override
	public boolean equals(Object o){
		if (o instanceof Node){
			return equals((Node) o);
		}
		return false;
	}
	/** Print out tree (with prices) */
	public void printAdv(){
		printAdv(0);
	}
	protected void printAdv(int nLevel){
		String sSpace = "";
		for (int a = 0; a < nLevel; a++){
			sSpace = sSpace + "	";
		}
		String sOutput = "";
		if (this.getClass().equals(Cargo.class)){
				sOutput = " "
				+ " MaxP: "+((Cargo)this).getPriceMaximum()
				+ " AvgP: "+((Cargo)this).getPriceAverage()
				+ " MinP: "+((Cargo)this).getPriceMinimum()
				+ " MaxQ: "+((Cargo)this).getQuantityMaximum()
				+ " AvgQ: "+((Cargo)this).getQuantityAverage()
				+ " MinQ: "+((Cargo)this).getQuantityMinimum();
		}
		System.out.println(sSpace+this.getName()+sOutput);
		Enumeration<String> keys = nodes.keys();
		while (keys.hasMoreElements()){
			nodes.get(keys.nextElement()).printAdv(nLevel+1);
		}
	}
	/** Print out tree (without prices) */
	public void print(){
		print(0, "");
	}
	protected void print(int nLevel, String sPath){
		if (nLevel != 0){
			sPath = sPath+"/"  + this.getName(); // and itself to the path
		} else {
			sPath = this.getName(); //Root, and itself
		}
		Enumeration<String> keys = nodes.keys(); //Get values for nodes
		while (keys.hasMoreElements()){
			nodes.get(keys.nextElement()).print(nLevel+1, sPath); //Call children's println
		}
		if (this.getClass().equals(Cargo.class)) System.out.println(sPath); //Print the path
	}
	/** Compare this node to <code>o</code>
	 * @param o         Object to compare to
	 * @return          the value <code>0</code> if the argument node is equal to
     *                  this node; a value less than <code>0</code> if this node
     *                  is lexicographically less than the node argument; and a
     *                  value greater than <code>0</code> if this node is
     *                  lexicographically greater than the node argument.
	 */
    @Override
	public int compareTo(Node o) {
		return this.getName().compareTo(o.getName());
	}
	/** return the name of this node
	 * @return          The name of this node
	 */
    @Override
	public String toString(){
		return sName;
	}
	public void clear(){
		Enumeration keys = nodes.keys();
		while (keys.hasMoreElements()){
			nodes.get(keys.nextElement()).clear();
		}
		nodes.clear();
		values.clear();
		
		nodes = null;
		values = null;
	}
}
