/*
 * Program.java - This file is part of VSTrade.
 *
 * Copyright (C) 2007 Niklas Kyster Rasmussen
 *
 * VSTrade is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * VSTrade is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VSTrade; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * FILE DESCRIPTION:
 * Create the GUI moduls, load/save data, and handle cross-modul interaction
 */

package vstrade;

//Java
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.Vector;
import javax.swing.JOptionPane;
import javax.swing.JTable;

//Glazed Lists
import ca.odell.glazedlists.swing.EventTableModel;

//VSTrade
import vstrade.data.settings.Settings;
import vstrade.moduls.status.StatusView;
import vstrade.moduls.about.AboutView;
import vstrade.moduls.cargo.CargoView;
import vstrade.moduls.frame.TabFrame;
import vstrade.moduls.map.MapView;
import vstrade.moduls.price.PriceView;
import vstrade.moduls.price.PriceTableRow;
import vstrade.moduls.settings.SettingsView;
import vstrade.moduls.trade.TradeView;
import vstrade.moduls.trade.TradeTableRow;
import vstrade.moduls.help.HelpView;
import vstrade.data.bases.Base;
import vstrade.data.bases.Bases;
import vstrade.data.bases.Cargo;
import vstrade.data.galaxy.Galaxy;
import vstrade.io.DataCreater;
import vstrade.io.DataSaver;
import vstrade.io.Log;
import vstrade.io.VsTradeUncaughtExceptionHandler;

public class Program implements ActionListener, MouseListener {
	/** ActionCommand: ADD PriceTab search base FROM TradeTab table buy base */
	public final static String ACTION_TRADE_PRICE_ADD_BUY_BASE = "TradePriceAddBuyBase";
	/** ActionCommand: ADD PriceTab search base FROM TradeTab table sell base */
	public final static String ACTION_TRADE_PRICE_ADD_SELL_BASE = "TradePriceAddSellBase";
	/** ActionCommand: SET PriceTab cargo FROM TradeTab table cargo  */
	public final static String ACTION_TRADE_PRICE_SET_CARGO = "TradePriceSetCargo";
	/** ActionCommand: SET CargoTab cargo FROM TradeTab table cargo  */
	public final static String ACTION_TRADE_CARGO_SET_CARGO = "TradeCargoSetCargo";
	/** ActionCommand: ADD TradeTab buy base FROM PriceTab table base */
	public final static String ACTION_PRICE_TRADE_ADD_AS_BUY_BASE = "PriceTradeAddAsBuyBase";
	/** ActionCommand: ADD TradeTab sell base FROM PriceTab table base */
	public final static String ACTION_PRICE_TRADE_ADD_AS_SELL_BASE = "PriceTradeAssAsSellBase";
	/** ActionCommand: SET CargoTab cargo FROM PriceTab table cargo */
	public final static String ACTION_PRICE_CARGO_SET_CARGO = "PriceCargoSetCargo";
	
	//Moduls
	private StatusView statusView;
	private TabFrame tabFrame;
	private TradeView tradeView;
	private PriceView priceView;
	private CargoView cargoView;
	private MapView mapView;
	private SettingsView settingsView;
	private AboutView aboutView;
	private HelpView helpView;
	
	//data
	private Settings settings = null;
	private Galaxy galaxy;
	private Bases bases;
	
	/** Creates a new instance of Program (Creates moduls and loaded extenal data) */
	public Program() {
		System.setProperty("sun.awt.exception.handler", "vstrade.io.VsTradeUncaughtExceptionHandler");
		Thread.setDefaultUncaughtExceptionHandler( new VsTradeUncaughtExceptionHandler());
		
		Log.info("Starting VSTrade 0.3.0.0");
	//Load Settings
		settings = new Settings();
		//Test if the VegaStrike directory is vaild (If not, try to find it)
		if (!settings.getFileSettings().isVegaStrikePathVaild() && settings.getAutoConfig()){
			Vector<String> sVsPaths = settings.getFileSettings().getValidVegaStrikePaths();
			if (sVsPaths.size() > 0){
				String s = (String)JOptionPane.showInputDialog(
					null,
					settings.getLanguageSettings().getString("SelectVSDir", "program"),
					settings.getLanguageSettings().getString("VSTradeConfig", "program"),
					JOptionPane.PLAIN_MESSAGE,
					null,
					sVsPaths.toArray(),
					sVsPaths.get(0));
				if (s != null){
					settings.getFileSettings().setVegaStrikePath(s);
					settings.saveSettings();
				}
			}
		}
		SplashUpdater.setProgress(10);
	//Create GUI
		
		SplashUpdater.setText( settings.getLanguageSettings().getString("LoadingGUI", "program") );
		tabFrame = new TabFrame();
		statusView = new StatusView(this);
		tabFrame.setStatusbar(statusView);
		SplashUpdater.setProgress(15);
		try {
			tradeView = new TradeView(this);
		} catch (NoClassDefFoundError e){
			if (e.toString().contains("glazedlists")){
				Log.error("The Glazed Lists library is missing (lib\\glazedlists-1.7.0_java15.jar)\nPlease see vstrade_faq.txt for more information.", e);
			} else {
				Log.error(e);
			}
		}
		tabFrame.addTab(tradeView);
		SplashUpdater.setProgress(20);
		
		priceView = new PriceView(this);
		tabFrame.addTab(priceView);
		SplashUpdater.setProgress(25);
		
		cargoView = new CargoView(this);
		tabFrame.addTab(cargoView);
		SplashUpdater.setProgress(30);
		
		mapView = new MapView(this);
		tabFrame.addTab(mapView);
		SplashUpdater.setProgress(35);
		
		settingsView = new SettingsView(this);
		tabFrame.addTab(settingsView);
		SplashUpdater.setProgress(45);
		
		aboutView = new AboutView(this);
		tabFrame.addTab(aboutView);
		
		//helpView = new HelpView(this);
		//tabFrame.addTab(helpView);
		SplashUpdater.setProgress(50);
		
		Log.info("GUI loaded");
		
		//Loading Bases and Galaxy
		SplashUpdater.setText( settings.getLanguageSettings().getString("LoadingData", "program") );
		clearData();
		DataCreater dataCreater = new DataCreater(this);
		dataCreater.run();
	}
	/** Set galaxy  */
	public void setGalaxy(Galaxy galaxy){
		this.galaxy = galaxy;
	}
	/** Set bases */
	public void setBases(Bases bases){
		this.bases = bases;
	}
	public Bases getBases(){
		return this.bases;
	}
	public Galaxy getGalaxy(){
		return this.galaxy;
	}
	public Settings getSettings(){
		return this.settings;
	}
	public StatusView getStatusView(){
		return this.statusView;
	}
	/** Load external data (in thread). */
	public void load(){
		clearData();
		DataCreater dataCreater = new DataCreater(this);
		dataCreater.start();
		//settingsUpdated();
	}
	/** Save settings and load external data (if needed). */
	public void save(){
		//Save settings
		DataSaver dataSaver = new DataSaver(this,  settings.getFileSettings().isVegaStrikePathUpdated());
		dataSaver.start();
	}
	/** Parse data to moduls and update the language. */
	public void setData(){
		statusView.setData();
		mapView.setData();
		priceView.setData();
		tradeView.setData();
		cargoView.setData();
		settingsView.setData();
		aboutView.setData();
		//helpView.setData();
		tabFrame.setData();
		tabFrame.show();
		System.gc();
	}
	public void clearData(){
		if (bases != null) {
			bases.clear();
			bases = null;
		}
		if (galaxy != null) {
			galaxy.clear();
			galaxy = null;
		}
		statusView.clearData();
		mapView.clearData();
		priceView.clearData();
		tradeView.clearData();
		cargoView.clearData();
		settingsView.clearData();
		aboutView.clearData();
		//helpView.clearData();
		tabFrame.clearData();
		System.gc();
	}
	/** ActionListener */
	@Override
	public void actionPerformed(ActionEvent e) {
		if (ACTION_TRADE_PRICE_ADD_BUY_BASE.equals(e.getActionCommand())) {
			Base base = tradeView.getSelectedBuyBase();
			if (base == null) return;
			priceView.addBase(base);
			if (settings.getSwitchComponent()) tabFrame.setSelectedComponent(priceView);
		}
		if (ACTION_TRADE_PRICE_ADD_SELL_BASE.equals(e.getActionCommand())) {
			Base base = tradeView.getSelectedSellBase();
			if (base == null) return;
			priceView.addBase(base);
			if (settings.getSwitchComponent()) tabFrame.setSelectedComponent(priceView);
		}
		if (ACTION_TRADE_PRICE_SET_CARGO.equals(e.getActionCommand())) {
			Cargo cargo = tradeView.getSelectedCargo();
			if (cargo == null) return;
			priceView.setSelectedCargo(cargo);
			if (settings.getSwitchComponent()) tabFrame.setSelectedComponent(priceView);
		}
		if (ACTION_TRADE_CARGO_SET_CARGO.equals(e.getActionCommand())) {
			Cargo cargo = tradeView.getSelectedCargo();
			if (cargo == null) return;
			cargoView.setSelectedCargo(cargo);
			if (settings.getSwitchComponent()) tabFrame.setSelectedComponent(cargoView);
		}
		if (ACTION_PRICE_TRADE_ADD_AS_BUY_BASE.equals(e.getActionCommand())) {
			Base base = priceView.getSelectedBase();
			if (base == null) return;
			tradeView.addBuyBase(base);
			if (settings.getSwitchComponent()) tabFrame.setSelectedComponent(tradeView);
		}
		if (ACTION_PRICE_TRADE_ADD_AS_SELL_BASE.equals(e.getActionCommand())) {
			Base base = priceView.getSelectedBase();
			if (base == null) return;
			tradeView.addSellBase(base);
			if (settings.getSwitchComponent()) tabFrame.setSelectedComponent(tradeView);
		}
		if (ACTION_PRICE_CARGO_SET_CARGO.equals(e.getActionCommand())) {
			Cargo cargo = priceView.getSelectedCargo();
			if (cargo == null) return;
			cargoView.setSelectedCargo(cargo);
			if (settings.getSwitchComponent()) tabFrame.setSelectedComponent(cargoView);
		}
	}
	/** MouseListener */
	@Override
	public void mousePressed(MouseEvent e) { }
	/** MouseListener */
	@Override
	public void mouseReleased(MouseEvent e) { }
	/** MouseListener */
	@Override
	public void mouseEntered(MouseEvent e) { }
	/** MouseListener */
	@Override
	public void mouseExited(MouseEvent e) { }
	/** MouseListener */
	@Override
	public void mouseClicked(MouseEvent e) {
		 //Double click on table (Switch to Cargo, and select clicked cargo)
		if (e.getClickCount() == 2 && e.getButton() == MouseEvent.BUTTON1) {
			Object o = e.getSource();
			if (o instanceof JTable){
				JTable jTable = (JTable) o;
				int row = jTable.getSelectedRow();
				if (row < 0) return;
				EventTableModel jTableModel = (EventTableModel) jTable.getModel();
				o = jTableModel.getElementAt(row);
				Cargo cargo = null;
				if (o instanceof TradeTableRow){ //Double click on Trade
					TradeTableRow tradeTableRow = (TradeTableRow) o;
					cargo = tradeTableRow.getCargo();
				}
				if (o instanceof PriceTableRow){ //Double click on Price
					PriceTableRow priceTableRow = (PriceTableRow) o;
					cargo = priceTableRow.getCargo();
				}
				if (cargo != null){ //Switch to cargo and select the clicked cargo
					cargoView.setSelectedCargo(cargo);
					tabFrame.setSelectedComponent(cargoView);
				}

			}
		}
	}
}
